package dynamo

import (
	"fmt"
	"strconv"
	"time"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/juju/errgo"
	log "github.com/sirupsen/logrus"
)

func attributeMapContains(attributeMap map[string]*dynamodb.AttributeValue, attributeName string) bool {
	_, exists := attributeMap[attributeName]
	return exists
}

func StringFromAttributes(attributeMap map[string]*dynamodb.AttributeValue, attributeName string) string {
	var result string
	attribute, exists := attributeMap[attributeName]
	if exists {
		result = *attribute.S
	}
	return result
}

func OptionalStringFromAttributes(attributeMap map[string]*dynamodb.AttributeValue, attributeName string) *string {
	if !attributeMapContains(attributeMap, attributeName) {
		return nil
	}
	attr := StringFromAttributes(attributeMap, attributeName)
	return &attr
}

func BoolFromAttributes(attributeMap map[string]*dynamodb.AttributeValue, attributeName string) bool {
	var result bool
	attribute, exists := attributeMap[attributeName]
	if exists {
		result = *attribute.BOOL
	}
	return result
}

func OptionalBoolFromAttributes(attributeMap map[string]*dynamodb.AttributeValue, attributeName string) *bool {
	if !attributeMapContains(attributeMap, attributeName) {
		return nil
	}
	attr := BoolFromAttributes(attributeMap, attributeName)
	return &attr
}

func Int64FromAttributes(attributeMap map[string]*dynamodb.AttributeValue, attributeName string) (int64, error) {
	var result int64
	attribute, exists := attributeMap[attributeName]
	if exists {
		// Do not use := to create err, will also create new result in narrower scope
		var err error
		result, err = strconv.ParseInt(*attribute.N, 10, 64)
		if err != nil {
			msg := fmt.Sprintf("Error converting dynamo attribute %s to int64", attributeName)
			log.Errorf(msg)
			return result, errgo.Notef(err, msg)
		}
	}
	return result, nil
}

func OptionalInt64FromAttributes(attributeMap map[string]*dynamodb.AttributeValue, attributeName string) (*int64, error) {
	if !attributeMapContains(attributeMap, attributeName) {
		return nil, nil
	}
	attr, err := Int64FromAttributes(attributeMap, attributeName)
	if err != nil {
		return nil, err
	}
	return &attr, nil
}

func TimeFromAttributesInSeconds(attributeMap map[string]*dynamodb.AttributeValue, attributeName string) (time.Time, error) {
	var result time.Time
	attribute, exists := attributeMap[attributeName]
	if exists {
		unixResult, err := strconv.ParseInt(*attribute.N, 10, 64)
		if err != nil {
			msg := fmt.Sprintf("Error converting dynamo attribute %s to time", attributeName)
			log.Errorf(msg)
			return result, errgo.Notef(err, msg)
		}
		result = time.Unix(unixResult, 0)
	}
	return result, nil
}

func TimeFromAttributes(attributeMap map[string]*dynamodb.AttributeValue, attributeName string) (time.Time, error) {
	var result time.Time
	attribute, exists := attributeMap[attributeName]
	if exists {
		unixNanoResult, err := strconv.ParseInt(*attribute.N, 10, 64)
		if err != nil {
			msg := fmt.Sprintf("Error converting dynamo attribute %s to string", attributeName)
			log.Errorf(msg)
			return result, errgo.Notef(err, msg)
		}
		result = time.Unix(0, unixNanoResult)
	}
	return result, nil
}

func PopulateAttributesString(attributeMap map[string]*dynamodb.AttributeValue, attributeName string, attributeValue string) {
	if attributeValue != "" {
		attributeMap[attributeName] = &dynamodb.AttributeValue{S: aws.String(attributeValue)}
	}
}

func PopulateAttributesBool(attributeMap map[string]*dynamodb.AttributeValue, attributeName string, attributeValue bool) {
	attributeMap[attributeName] = &dynamodb.AttributeValue{BOOL: aws.Bool(attributeValue)}
}

func PopulateAttributesInt64(attributeMap map[string]*dynamodb.AttributeValue, attributeName string, attributeValue int64) {
	strVal := strconv.FormatInt(attributeValue, 10)
	attributeMap[attributeName] = &dynamodb.AttributeValue{N: aws.String(strVal)}
}

func PopulateAttributesTime(attributeMap map[string]*dynamodb.AttributeValue, attributeName string, time time.Time) {
	if time.UnixNano() > 0 {
		unixNanoStr := strconv.FormatInt(time.UnixNano(), 10)
		attributeMap[attributeName] = &dynamodb.AttributeValue{N: aws.String(unixNanoStr)}
	}
}

func NewPutItemInput(tableName string, record DynamoTableRecord) *dynamodb.PutItemInput {
	record.GetTable()
	itemMap := record.NewAttributeMap()
	return &dynamodb.PutItemInput{
		TableName: aws.String(tableName),
		Item:      itemMap,
	}
}

func NewUpdateItemInput(tableName string, record DynamoTableRecord) *dynamodb.UpdateItemInput {
	itemMap := record.NewAttributeMap()
	key := record.NewItemKey()
	return &dynamodb.UpdateItemInput{
		TableName:        aws.String(tableName),
		Key:              key,
		AttributeUpdates: AttributeMapToAttributeUpdateMap(key, itemMap),
	}
}

func NewGetItemInput(tableName string, record DynamoTableRecord) *dynamodb.GetItemInput {
	return &dynamodb.GetItemInput{
		TableName: aws.String(tableName),
		Key:       record.NewItemKey(),
	}
}

func NewDeleteItemInput(tableName string, record DynamoTableRecord) *dynamodb.DeleteItemInput {
	return &dynamodb.DeleteItemInput{
		TableName: aws.String(tableName),
		Key:       record.NewItemKey(),
	}
}

func attributeMapToAttributeUpdateMapWithAction(key map[string]*dynamodb.AttributeValue, attributeMap map[string]*dynamodb.AttributeValue, action string) map[string]*dynamodb.AttributeValueUpdate {
	attributeUpdateMap := make(map[string]*dynamodb.AttributeValueUpdate)
	for attr, attrVal := range attributeMap {
		_, isKey := key[attr]
		if !isKey && attrVal != nil {
			attrUpdate := &dynamodb.AttributeValueUpdate{
				Action: aws.String(action),
				Value:  attrVal,
			}
			attributeUpdateMap[attr] = attrUpdate
		}
	}
	return attributeUpdateMap
}

func AttributeMapToAttributeUpdateMap(key map[string]*dynamodb.AttributeValue, attributeMap map[string]*dynamodb.AttributeValue) map[string]*dynamodb.AttributeValueUpdate {
	return attributeMapToAttributeUpdateMapWithAction(key, attributeMap, "PUT")
}

func AttributeMapToAttributeUpdateMapAdd(key map[string]*dynamodb.AttributeValue, attributeMap map[string]*dynamodb.AttributeValue) map[string]*dynamodb.AttributeValueUpdate {
	return attributeMapToAttributeUpdateMapWithAction(key, attributeMap, "ADD")
}
