package integration_test

import (
	"math/rand"
	"net/http"
	"strconv"
	"testing"

	"code.justin.tv/foundation/twitchclient"
	samus_gateway "code.justin.tv/samus/gateway/client"
	"code.justin.tv/samus/gateway/clients"
	nitro "code.justin.tv/samus/nitro/rpc"
	uuid "github.com/satori/go.uuid"
	. "github.com/smartystreets/goconvey/convey"
	"golang.org/x/net/context"
)

// beta-endpoint gateway client
var conf = twitchclient.ClientConf{
	Host: "staging-samus-gateway-env.ybgep8dqmz.us-west-2.elasticbeanstalk.com",
}

var client, _ = samus_gateway.NewClient(conf)

var nitroConf = twitchclient.ClientConf{
	Host:           "staging-nitro.us-west-2.elasticbeanstalk.com",
	TimingXactName: "Nitro-Client",
	Stats:          conf.Stats,
}

var nitroClient = nitro.NewNitroProtobufClient(nitroConf.Host, twitchclient.NewHTTPClient(nitroConf))
var tproxClient, _ = clients.NewSamusTProxClient()

// Default test users
// https://w.amazon.com/bin/view/Samus/Tech/HUD/Runbooks#Test_Users
const (
	ellaMozzarella       = "250256873"
	ivoneProvolone       = "207375363"
	marieManouri         = "450819272" // user with date+country override whitelist in SOS for Beta and Prod
	pierreGruyere        = "207105242"
	unlinkedUserId       = "101010101"
	defaultTestChannelId = "151269150" // qa_auto_subs_partner1

	marieManouriEcid = "A1KA7N3H4DDG8I"

	// An offer in prod claimed by marieManouri. This is the only existing offer in prod for the [dateOverrideForClaimedOffer] timestamp.
	// Although all our integ tests run against beta, the TProx apis such as [GetCurrentOffersForUser] must user prod data since there is only a prod TProx endpoint.
	claimedProdOfferID          = "32ad5a54-6565-6920-74fb-eacc0a27c931"
	dateOverrideForClaimedOffer = "6050-10-14T12:00:01Z"

	// An EXTERNAL offer in prod unclaimed by marieManouri. This is the only existing offer in prod for the [dateOverrideForUnclaimedExternalOffer] timestamp.
	unclaimedExternalProdOfferID          = "ecb15dd9-c713-2994-c219-5da2f96f7c32"
	dateOverrideForUnclaimedExternalOffer = "6048-04-20T12:00:01Z"

	// A DIRECT_ENTITLEMENT offer in prod unclaimed by marieManouri. This is the only existing offer in prod for the [dateOverrideForUnclaimedOffer] timestamp.
	unclaimedProdOfferID          = "68b9537f-e627-1b9d-333e-ddc2d40988cd"
	dateOverrideForUnclaimedOffer = "6150-10-14T16:01:00Z"

	// An offer in beta. This is the only existing offer in prod for the [dateOverrideForBetaOffer] timestamp.
	betaOfferID              = "f0b627b8-7942-eece-9f35-cdda508cf476"
	dateOverrideForBetaOffer = "6050-10-14T12:00:01Z"

	// Order placed in beta, by marieManouri's linked amazon account (marieManouriEcid)
	betaOrderID = "amzn1.pg.order.d486400b-ded5-47d0-8dbe-51d7cd6cb00b"

	catalogBetaOfferID              = "amzn1.pg.offer.40bb6fbe-c436-f208-b15c-d19df12f901b"
	dateOverrideForCatalogBetaOffer = "6050-10-14T12:00:01Z"

	// country codes for the aforementioned offers
	validCountryCodeForOffer   = "US"
	invalidCountryCodeForOffer = "SA"
)

func TestDynamicStringsApi(t *testing.T) {
	Convey("Testing strings api ", t, func() {
		userId := pierreGruyere
		r, err := client.GetDynamicStrings(context.Background(), userId, []string{"twitch.prime.bluebar.left1"}, "US", "en", "", nil)

		So(err, ShouldBeNil)
		So(r.DynamicStringMap["twitch.prime.bluebar.left1"].ID, ShouldEqual, "twitch.prime.bluebar.left1")
		So(r.DynamicStringMap["twitch.prime.bluebar.left1"].ExternalUrl, ShouldEqual, "https://twitch.amazon.com/prime?ref_=sm_w_thp_blue_tw_all")
		So(r.DynamicStringMap["twitch.prime.bluebar.left1"].Text, ShouldEqual, "Twitch Prime")
		So(r.DynamicStringMap["twitch.prime.bluebar.left1"].IsExternalLink, ShouldEqual, true)
	})
}

func TestPrimeStatusApi(t *testing.T) {
	Convey("Testing user status API ", t, func() {
		userId := pierreGruyere
		r, err := client.GetPrimeStatus(context.Background(), userId, nil)

		So(err, ShouldBeNil)
		So(r.TwitchPrime, ShouldEqual, true)
		So(r.UserID, ShouldEqual, userId)
	})

	Convey("Testing Nitro's status API ", t, func() {
		userId := pierreGruyere
		req := &nitro.GetPremiumStatusesRequest{
			TwitchUserID: userId,
		}
		r, err := nitroClient.GetPremiumStatuses(context.Background(), req)

		So(err, ShouldBeNil)
		So(r.HasPrime, ShouldEqual, true)
	})
}

func TestGetPrimeSettingsApi(t *testing.T) {
	Convey("Testing GetPrimeSettings API", t, func() {
		Convey("with an existent prime user", func() {
			userId := pierreGruyere
			resp, err := client.GetPrimeSettings(context.Background(), userId, nil)

			So(err, ShouldBeNil)
			So(resp.UserID, ShouldEqual, userId)
			So(resp.IsSubscriptionMessage, ShouldBeTrue)
		})
	})
}

func TestSetPrimeSettingsApi(t *testing.T) {
	Convey("Testing SetPrimeSettings API", t, func() {
		userId := pierreGruyere
		Convey("setting subscription notification to false", func() {
			settings := samus_gateway.PrimeSettings{
				IsSubscriptionMessage: false,
			}

			resp, err := client.SetPrimeSettings(context.Background(), userId, settings, nil)

			So(err, ShouldBeNil)
			So(resp.UserID, ShouldEqual, userId)
			So(resp.IsSubscriptionMessage, ShouldBeFalse)
		})

		Convey("setting subscription notification to true", func() {
			settings := samus_gateway.PrimeSettings{
				IsSubscriptionMessage: true,
			}

			resp, err := client.SetPrimeSettings(context.Background(), userId, settings, nil)

			So(err, ShouldBeNil)
			So(resp.UserID, ShouldEqual, userId)
			So(resp.IsSubscriptionMessage, ShouldBeTrue)
		})
	})
}

func TestGetBalance(t *testing.T) {
	Convey("Testing GetBalance API ", t, func() {
		userId := pierreGruyere
		r, err := client.GetBalance(context.Background(), userId, nil)

		So(err, ShouldBeNil)
		So(r.SubscriptionCreditBalance, ShouldBeBetweenOrEqual, 0, 1)
	})
}

func TestCanSpendPrimeCredit(t *testing.T) {
	Convey("Testing CanSpendPrimeCredit API ", t, func() {
		userId := pierreGruyere
		productId := defaultTestChannelId
		_, err := client.CanSpendPrimeCredit(context.Background(), userId, productId, nil)

		So(err, ShouldBeNil)
		// TODO - update when channel is actually used
	})
}

// Note: Currently failing. Waiting on backend implementation.
func TestSpendSubscriptionCredit(t *testing.T) {
	Convey("Testing SpendSubscriptionCredit API", t, func() {
		Convey("with a non Prime user", func() {
			userId := ivoneProvolone
			broadcasterId := defaultTestChannelId
			resp, errorCode, err := client.SpendSubscriptionCredit(context.Background(), userId, broadcasterId, nil)

			So(err, ShouldBeNil)
			So(resp, ShouldBeNil)
			So(errorCode, ShouldNotBeNil)

			So(errorCode.Code, ShouldEqual, samus_gateway.NoBalance)
		})

		Convey("with a Prime user without credit", func() {
			userId := pierreGruyere
			broadcasterId := defaultTestChannelId
			resp, errorCode, err := client.SpendSubscriptionCredit(context.Background(), userId, broadcasterId, nil)

			So(err, ShouldBeNil)
			So(resp, ShouldBeNil)
			So(errorCode, ShouldNotBeNil)
			So(errorCode.Code, ShouldEqual, samus_gateway.TooManyRecentSpends)
		})

		/**
		 * NOTE: It's not possible currently to test the SpendSubscriptionCredit API locally/in Beta reliably since
		 * Kamino accounts are recycled thus unlinking Twitch accounts. The test below should fail in these stages, but
		 * should pass in Prod.
		 */
		SkipConvey("with a Prime user with credit", func() {
			userId := ellaMozzarella
			broadcasterId := defaultTestChannelId
			spendResp, errorCode, err := client.SpendSubscriptionCredit(context.Background(), userId, broadcasterId, nil)

			So(err, ShouldBeNil)
			So(errorCode, ShouldBeNil)
			So(spendResp, ShouldNotBeNil)
			So(spendResp.UserID, ShouldEqual, userId)
			So(spendResp.BroadcasterID, ShouldEqual, broadcasterId)
			So(spendResp.SubscriptionCreditBalance, ShouldEqual, 0)

			// Unspend to reset the credit
			errorCode, err = client.UnspendSubscriptionCredit(context.Background(), userId, broadcasterId, nil)
			So(err, ShouldBeNil)
			So(errorCode, ShouldBeNil)

			// Check that the credit was restored to 1
			balanceResp, err := client.GetBalance(context.Background(), userId, nil)
			So(err, ShouldBeNil)
			So(balanceResp, ShouldNotBeNil)
			So(balanceResp.SubscriptionCreditBalance, ShouldEqual, 1)
		})
	})
}

func TestGetPrimeOfferEntitlement(t *testing.T) {
	Convey("Testing GetPrimeOfferEntitlement API ", t, func() {
		userId := pierreGruyere
		offerId := "d2ad6ed1-93ee-fd99-86db-907dd4a950d0"
		r, err := client.GetPrimeOfferEntitlement(context.Background(), userId, offerId, "en", nil)

		So(err, ShouldBeNil)
		So(r.UserID, ShouldEqual, userId)
		So(r.OfferID, ShouldEqual, offerId)
		So(r.HasEntitlement, ShouldEqual, true)
	})
}

func TestAccountLinking(t *testing.T) {
	Convey("Account Linking API", t, func() {
		var toString = strconv.Itoa
		userID := "00" + toString(rand.Int())
		gameName := "callofduty"
		gameAccount := samus_gateway.GameAccount{
			AccountID:   toString(rand.Int()),
			DisplayName: toString(rand.Int()),
		}

		link := samus_gateway.GameAccountLink{
			UserID:      userID,
			GameAccount: gameAccount,
		}

		Convey("With a linked account", func() {
			client.CreateAccountLink(context.Background(), userID, gameName, gameAccount, nil)

			Convey("Cannot link again", func() {
				_, status, _ := client.CreateAccountLink(context.Background(), userID, gameName, gameAccount, nil)
				So(status, ShouldEqual, http.StatusConflict)
			})

			Convey("Can read the link", func() {
				r, err := client.GetAccountLink(context.Background(), userID, gameName, nil)
				So(err, ShouldBeNil)
				So(r.AccountLink, ShouldResemble, link)
			})

			Convey("Can delete the link", func() {
				r, err := client.DeleteAccountLink(context.Background(), userID, gameName, nil)
				So(err, ShouldBeNil)
				So(r.Deleted, ShouldNotBeNil)
				So(*r.Deleted, ShouldResemble, link)
			})

			client.DeleteAccountLink(context.Background(), userID, gameName, nil)
		})

		Convey("Without a link", func() {
			client.DeleteAccountLink(context.Background(), userID, gameName, nil)

			Convey("Cannot read the link", func() {
				_, err := client.GetAccountLink(context.Background(), userID, gameName, nil)
				So(err, ShouldNotBeNil)
				So(err.(*twitchclient.Error).StatusCode, ShouldEqual, http.StatusNotFound)
			})

			Convey("Can attempt to delete the link", func() {
				r, err := client.DeleteAccountLink(context.Background(), userID, gameName, nil)
				So(err, ShouldBeNil)
				So(r.Deleted, ShouldBeNil)
			})

			Convey("Can create a link", func() {
				r, status, err := client.CreateAccountLink(context.Background(), userID, gameName, gameAccount, nil)
				So(err, ShouldBeNil)
				So(status, ShouldEqual, http.StatusOK)
				So(r.AccountLink, ShouldResemble, link)

				client.DeleteAccountLink(context.Background(), userID, gameName, nil)
			})
		})
	})
}

func TestClearOfferClaimCodeForUser(t *testing.T) {
	Convey("Testing ClearOfferClaimCodeForUser API", t, func() {
		Convey("with success", func() {
			userId := pierreGruyere
			offerId := claimedProdOfferID
			r, _, err := client.ClearOfferClaimCodeForUser(context.Background(), userId, offerId, nil)

			So(err, ShouldBeNil)
			So(r.Success, ShouldBeTrue)
		})
	})
}

// TODO: Get new beta offer IDs for Catalog integration
func TestGetCurrentOffersWithEligibility(t *testing.T) {
	userId := marieManouri

	Convey("Testing GetCurrentOffersWithEligibility API", t, func() {
		Convey("GetCurrentOffersWithEligibility: Returns offers when country code and date override parameters are not used", func() {
			resp, err := client.GetCurrentOffersWithEligibility(context.Background(), userId, "", "en", "", nil)
			So(err, ShouldBeNil)
			So(resp.CurrentOffers, ShouldNotBeEmpty)
		})

		Convey("GetCurrentOffersWithEligibility: Returns correct offers when country code and date override parameters are used", func() {
			resp, err := client.GetCurrentOffersWithEligibility(context.Background(), userId, validCountryCodeForOffer, "en", dateOverrideForCatalogBetaOffer, nil)
			So(err, ShouldBeNil)
			So(resp.CurrentOffers, ShouldNotBeEmpty)
			So(resp.CurrentOffers[0].CatalogOfferID, ShouldEqual, catalogBetaOfferID)
		})

		Convey("GetCurrentOffersWithEligibility: Does not return incorrect offers when country code and date override parameters are used", func() {
			resp, err := client.GetCurrentOffersWithEligibility(context.Background(), userId, invalidCountryCodeForOffer, "en", dateOverrideForCatalogBetaOffer, nil)
			So(err, ShouldBeNil)
			So(resp.CurrentOffers, ShouldBeEmpty)
		})
	})
}

func TestPlaceOrder(t *testing.T) {
	userId := marieManouri
	offerId := betaOfferID
	attributionChannel := "{test:test}"
	idempotenceKey := uuid.NewV4().String()

	Convey("Testing PlaceOrder API", t, func() {
		Convey("PlaceOrder: Returns orderId for an offer claim", func() {
			resp, errResp, err := client.PlaceOrder(context.Background(), userId, offerId, idempotenceKey, attributionChannel, "", nil)
			So(err, ShouldBeNil)
			So(errResp, ShouldBeNil)
			So(resp.OrderID, ShouldNotBeEmpty)
		})
	})
}

func TestGetOrdersByCustomer(t *testing.T) {
	userId := marieManouri
	pageSize := 10
	offerId := catalogBetaOfferID
	orderId := betaOrderID
	nextToken := "test"

	Convey("Testing GetOrdersByCustomer API", t, func() {
		Convey("GetOrdersByCustomer: Returns orders placed by the customer", func() {
			resp, errResp, err := client.GetOrdersByCustomer(context.Background(), userId, nextToken, pageSize, offerId, orderId, nil)
			So(err, ShouldBeNil)
			So(errResp, ShouldBeNil)
			So(resp.OrderDocuments, ShouldNotBeEmpty)
		})
	})
}

func TestListInventory(t *testing.T) {
	userId := marieManouri
	amazonCustomerId := marieManouriEcid
	entitlementAccountType := "AMAZON"
	nextToken := ""
	maxResults := 10
	itemIds := []string{}
	entitlementStatusFilters := []string{}

	Convey("Testing ListInventory API", t, func() {
		Convey("ListInventory: Returns inventory for the customer", func() {
			resp, errResp, err := client.ListInventory(context.Background(), userId, amazonCustomerId, entitlementAccountType, itemIds, nextToken, maxResults, entitlementStatusFilters, nil)
			So(err, ShouldBeNil)
			So(errResp, ShouldBeNil)
			So(resp.Inventory, ShouldNotBeEmpty)
		})
	})
}
