package main

import (
	log "github.com/sirupsen/logrus"

	"os"

	"code.justin.tv/common/config"
	"code.justin.tv/common/twitchhttp"
	"code.justin.tv/samus/gateway/api"
	"code.justin.tv/samus/gateway/backend"
	"code.justin.tv/video/roll"
	"code.justin.tv/video/rollrus"

	"strconv"
	"time"

	samus_config "code.justin.tv/samus/gateway/configuration"
	"code.justin.tv/samus/gateway/metrics"
	"code.justin.tv/samus/gateway/settings"
	"code.justin.tv/samus/gateway/workers"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
)

const WORKER_WAIT_TIMEOUT = 20 * time.Second

func init() {
	log.SetLevel(log.DebugLevel)
	log.Info("Starting samus-gateway")

	config.Register(settings.LoadConfig())
	err := config.Parse()
	if err != nil {
		log.Fatal(err)
	}

	log.Info("OS ENVIRONMENT is set to :", os.Getenv("ENVIRONMENT"))

	client := roll.New(config.Resolve(settings.RollbarToken), os.Getenv("ENVIRONMENT"))
	log.AddHook(rollrus.NewHook(client, log.ErrorLevel))
	log.StandardLogger().SetNoLock()
}

func main() {

	bConf := backend.BackendConf{
		Stats:                      config.Statsd(),
		SamusSWSHost:               config.Resolve(settings.SamusSWSHost),
		SamusSWSCertPath:           config.Resolve(settings.SamusSWSCertPath),
		RootCertPath:               config.Resolve(settings.RootCertPath),
		RedisHostPort:              config.Resolve(settings.RedisHostPort),
		RailsHost:                  config.Resolve(settings.RailsHost),
		DynamoRegion:               config.Resolve(settings.DynamoRegion),
		UserDynamoTableName:        config.Resolve(settings.UserDynamoTableName),
		EntitlementDynamoTableName: config.Resolve(settings.EntitlementDynamoTableName),
		PromoStringDynamoTableName: config.Resolve(settings.PromoStringDynamoTableName),
		AccountLinkDynamoTableName: config.Resolve(settings.AccountLinkDynamoTableName),
		SnsRegion:                  config.Resolve(settings.SnsRegion),
		PrimeStatusSnsTopicArn:     config.Resolve(settings.PrimeStatusSnsTopicArn),
		NitroEndpoint:              config.Resolve(settings.NitroEndpoint),
		OfferCacheExpiration:       config.Resolve(settings.OfferCacheExpiration),
		PubSubHost:                 config.Resolve(settings.PubSubHost),
	}

	backendApi, err := backend.NewBackend(&bConf)
	retries := 0
	for err != nil && retries < 3 {
		log.WithError(err)
		time.Sleep(5000 * time.Millisecond)
		retries++
		backendApi, err = backend.NewBackend(&bConf)
	}

	if err != nil {
		log.Info(err)
		log.WithError(err)
		panic("Unable to instantiate backend.")
	}

	metricsSession, err := session.NewSession(&aws.Config{
		Region: aws.String(settings.MetricsRegion),
	})

	metricsConfig := metrics.MetricsConfig{
		Environment:        config.Resolve(settings.Environment),
		Hostname:           config.Resolve(settings.Hostname),
		ServiceName:        config.Resolve(settings.ServiceName),
		MetricsRegion:      config.Resolve(settings.MetricsRegion),
		Marketplace:        config.Resolve(settings.Marketplace),
		MetricsCredentials: metricsSession.Config.Credentials,
	}

	samusConfigs := samus_config.SamusConfigurations{
		MetricsConfig: &metricsConfig,
		BackendConfig: &bConf,
	}

	server, err := api.NewServer(config.Statsd(), config.RollbarErrorLogger(), backendApi, samusConfigs)
	if err != nil {
		log.WithError(err)
	}

	numWorkers, err := strconv.Atoi(config.Resolve(settings.NumWorkersForSamusSQS))
	if err != nil {
		log.WithError(err)
	}
	if numWorkers > 0 {
		go workers.InitSQSWorkersForSamus(config.Statsd(), backendApi, numWorkers, config.Resolve(settings.SamusStatusQueueName))
	}

	numWorkers, err = strconv.Atoi(config.Resolve(settings.NumWorkersForOrderSQS))
	if err != nil {
		log.WithError(err)
	}
	if numWorkers > 0 {
		go workers.InitSQSWorkersForOrders(config.Statsd(), backendApi, numWorkers, config.Resolve(settings.SamusOrdersQueueName))
	}

	log.Fatal(twitchhttp.ListenAndServe(server))
}
