package metrics

import (
	"time"
	"strings"

	"code.justin.tv/foundation/twitchclient"
	"code.justin.tv/samus/AmazonMWSGoClient/mws"
	log "github.com/sirupsen/logrus"
)

const MS_PER_SECOND = 1000.0
const UNIT_ONE = 1.0
const COUNT = "Count"
const AVAILABILITY = "Availability"
const SECURE_REQUEST = "SecureRequest"
const INSECURE_REQUEST = "InsecureRequest"
const NONE = "NONE"
const ALL = "ALL"

const (
	hostSpecific  = false
	serviceName   = "SamusGateway"
	environment   = "SamusGateway/NA"
	hostname      = "AWS-ELB"
	partitionID   = ""
	customerID    = ""
	metricsRegion = "PDX" // PDX is us-west-2.

	minPeriod = mws.PeriodFiveMinute
	maxPeriod = mws.PeriodFiveMinute

	// Exported metrics types that can be used for counts.

	// MetricTypeSuccesses is the number of successes for a provided API.
	MetricTypeSuccesses = "Success"
	// MetricTypeServerError the number of server errors for a provided API.
	MetricTypeServerError = "Server_Error"
	// MetricTypeClientError the number of client errors for a provided API.
	MetricTypeClientError = "Client_Error"
)

// IMetricsClient interface
type IMetricsClient interface {
	// PostAvailabilityMetric emits % of successful requests
	PostAvailabilityMetric(methodName string, status int)
	// PostLatencyMetric emits latency metric for a method/API in UnitMilliseconds
	PostLatencyMetric(methodName string, duration time.Duration)
	// PostStatusMetric emits availability metric for a method/API in UnitMilliseconds
	PostStatusMetric(methodName string, status int)
	// PostDurationSinceMetric emits a specific duration for metric name provided since a specific time
	PostDurationSinceMetric(methodName string, timeOfEvent time.Time)
	// PostCountMetric emits a count for the metricName provided
	PostCountMetric(metricName string, value int)
	// Additional Helper APIs for Custom Metric Name
	// PostDecimalMetricWithMetricName - specify decimal metric of a specific metricName
	PostDecimalMetricWithMetricName(methodName string, decimal float64, metricName string)
	// PostLatencyMetricWithMetricName - specify latency of a specific metricName
	PostLatencyMetricWithMetricName(methodName string, latency time.Duration, metricName string)
	// PostAvailabilityAndLatencyMetrics
	PostAvailabilityAndLatencyMetrics(methodName string, status int, duration time.Duration)
	// PostSecureRequestMetrics
	PostSecureRequestMetrics(methodName string, protocol string)
}

// MetricLogger for the metrics, this object takes the actual metrics requests
type MetricsClient struct {
	config    *MetricsConfig
	mwsClient *mws.AmazonMWSGoClient
}

// NewMetricsClient creates a metrics client with MWS
func NewMetricsClient(config *MetricsConfig) (IMetricsClient, error) {
	endpoint := mws.Regions[config.MetricsRegion].ExternalEndpoint
	client, err := twitchclient.NewClient(twitchclient.ClientConf{
		TimingXactName: "MWS-Client",
		Host:           endpoint,
	})
	if err != nil {
		return nil, err
	}
	mwsClient := mws.NewAmazonMWSGoClient(client)
	return &MetricsClient{
		config:    config,
		mwsClient: mwsClient,
	}, nil
}

// PostAvailabilityMetric emits % of successful requests
func (metrics *MetricsClient) PostAvailabilityAndLatencyMetrics(methodName string, status int, latency time.Duration) {
	milliseconds := latency.Seconds() * MS_PER_SECOND
	success := calculateAvailability(status)
	availabilityMetric := metrics.createCountMetric(metrics.config, methodName, float64(success), AVAILABILITY)
	latencyMetric := metrics.createDecimalMetricWithMetricNameAndUnit(metrics.config, methodName, milliseconds, mws.MetricTime, mws.UnitMilliseconds)
	metricsToReport := []mws.Metric{availabilityMetric, latencyMetric}
	go metrics.postMetrics(metrics.config, metricsToReport)
}

// PostAvailabilityMetric emits % of successful requests
func (metrics *MetricsClient) PostAvailabilityMetric(methodName string, status int) {
	success := calculateAvailability(status)
	metrics.postCountMetric(metrics.config, methodName, float64(success), AVAILABILITY)
}

// PostLatencyMetric emits latency metric for a method/API in UnitMilliseconds
func (metrics *MetricsClient) PostLatencyMetric(methodName string, latency time.Duration) {
	milliseconds := latency.Seconds() * MS_PER_SECOND
	metrics.postDecimalMetricWithMetricNameAndUnit(metrics.config, methodName, milliseconds, mws.MetricTime, mws.UnitMilliseconds)
}

// PostStatusMetric emits availability metric for a method/API in UnitMilliseconds
func (metrics *MetricsClient) PostStatusMetric(methodName string, status int) {
	metrics.postCountMetric(metrics.config, methodName, UNIT_ONE, statusToMetricType(status))
}

// PostDurationSinceMetric emits a specific duration for metric name provided since a specific time
func (metrics *MetricsClient) PostDurationSinceMetric(methodName string, timeOfEvent time.Time) {
	latency := time.Now().Sub(timeOfEvent)
	milliseconds := latency.Seconds() * MS_PER_SECOND
	metrics.postDecimalMetricWithMetricNameAndUnit(metrics.config, methodName, milliseconds, mws.MetricTime, mws.UnitMilliseconds)
}

// PostVolumeMetric emits a count for the metricName provided
func (metrics *MetricsClient) PostCountMetric(metricName string, value int) {
	metrics.postCountMetric(metrics.config, ALL, float64(value), COUNT)
}

// PostDecimalMetricWithMetricName
func (metrics *MetricsClient) PostDecimalMetricWithMetricName(methodName string, decimal float64, metricName string) {
	metrics.postDecimalMetricWithMetricNameAndUnit(metrics.config, methodName, decimal, metricName, mws.MetricTime)
}

// PostLatencyMetricWithMetricName
func (metrics *MetricsClient) PostLatencyMetricWithMetricName(methodName string, latency time.Duration, metricName string) {
	metrics.postLatencyMetricWithMetricName(metrics.config, methodName, latency, metricName)
}

// PostSecureRequestMetrics emits metrics tracking requests forwarded from the load balancer from port 443 or port 80 
func (metrics *MetricsClient) PostSecureRequestMetrics(methodName string, protocol string) {
	if protocol != "" {
		secureRequest := determineSecureRequest(protocol)
		insecureRequest := determineInsecureRequest(protocol)
		secureRequestMetric := metrics.createCountMetric(metrics.config, methodName, float64(secureRequest), SECURE_REQUEST)
		insecureRequestMetric := metrics.createCountMetric(metrics.config, methodName, float64(insecureRequest), INSECURE_REQUEST)

		metricsToReport := []mws.Metric{secureRequestMetric, insecureRequestMetric}
		go metrics.postMetrics(metrics.config, metricsToReport)
	} else {
		log.Error("Invalid protocol for request, skipping metrics publish")
	}
}

func determineSecureRequest(protocol string) int {
	if strings.EqualFold(protocol, "https") {
		return 1
	}
	return 0
}

func determineInsecureRequest(protocol string) int {
	if strings.EqualFold(protocol, "http") {
		return 1
	}
	return 0
}

// statusToMetricType provides the metric instance name to use for an HTTP status code
func statusToMetricType(status int) string {
	var metricType = ""
	if status < 300 {
		// Anything below 300 is a success.
		metricType = MetricTypeSuccesses
	} else if status < 500 {
		// Anything between 300 and 500 is a client error of some sort.
		metricType = MetricTypeClientError
	} else {
		// Anything above 500 is a server error.
		metricType = MetricTypeServerError
	}

	return metricType
}

// calculateAvailability use status code to
func calculateAvailability(status int) int {
	if status < 500 {
		return 1
	}
	return 0
}

// PostLatencyMetricWithMetricName will post the latency for a given method and client to PMET, using a different name than Time
func (metrics *MetricsClient) postLatencyMetricWithMetricName(config *MetricsConfig, methodName string, latency time.Duration, metricName string) {
	milliseconds := latency.Seconds() * MS_PER_SECOND
	metrics.postDecimalMetricWithMetricNameAndUnit(config, methodName, milliseconds, metricName, mws.UnitMilliseconds)
}

// PostDecimalMetricWithMetricName will post a decimal metric for a given method and client to PMET, using the given metric name
func (metrics *MetricsClient) postDecimalMetricWithMetricName(config *MetricsConfig, methodName string, decimal float64, metricName string) {
	metrics.postDecimalMetricWithMetricNameAndUnit(config, methodName, decimal, metricName, COUNT)
}

// createDecimalMetricWithMetricNameAndUnit
func (metrics *MetricsClient) createDecimalMetricWithMetricNameAndUnit(config *MetricsConfig, methodName string, decimal float64, metricName string, unit string) mws.Metric {
	// Initialize the metric object.
	decimalMetric := mws.NewMetric(config.IsProd(), hostname, hostSpecific, config.GetAwsRegion(), serviceName, methodName, ALL, metricName)

	// Add the actual decimal value to the metric.
	decimalMetric.AddValue(decimal)
	decimalMetric.Unit = unit
	decimalMetric.Timestamp = time.Now().UTC()
	return decimalMetric
}

// createCountMetric
func (metrics *MetricsClient) createCountMetric(config *MetricsConfig, methodName string, value float64, metricType string) mws.Metric {
	// Initialize the metric
	countMetric := mws.NewMetric(config.IsProd(), hostname, hostSpecific, config.GetAwsRegion(), serviceName, methodName, ALL, metricType)

	// Add the success value.
	countMetric.AddValue(value)
	countMetric.Unit = COUNT
	countMetric.Timestamp = time.Now().UTC()

	return countMetric
}

// postDecimalMetricWithMetricNameAndUnit
func (metrics *MetricsClient) postDecimalMetricWithMetricNameAndUnit(config *MetricsConfig, methodName string, decimal float64, metricName string, unit string) {
	// Initialize the metric object.
	decimalMetric := metrics.createDecimalMetricWithMetricNameAndUnit(config, methodName, decimal, metricName, unit)

	// Post it !
	go metrics.postMetric(config, decimalMetric)
}

// postCountMetric posts a counter metric to PMET, the type is provided as a parameter.
func (metrics *MetricsClient) postCountMetric(config *MetricsConfig, methodName string, value float64, metricType string) {
	// Initialize the metric
	countMetric := metrics.createCountMetric(config, methodName, value, metricType)

	// Post it !
	go metrics.postMetric(config, countMetric)
}

func (metrics *MetricsClient) postMetric(config *MetricsConfig, metric mws.Metric) {
	if config.IsLocal() || config.IsTest() {
		// No metrics locally or for tests.
		return
	}
	metricsToReport := []mws.Metric{metric}
	metrics.postMetrics(config, metricsToReport)
}

func (metrics *MetricsClient) postMetrics(config *MetricsConfig, metricsToReport []mws.Metric) {
	if config.IsLocal() || config.IsTest() {
		// No metrics locally or for tests.
		return
	}

	// Initialize MetricReport and add the Metric.
	metricReport := mws.NewMetricReport(config.IsProd(), hostname, minPeriod, maxPeriod, partitionID, environment, customerID)
	for _, metric := range metricsToReport {
		metricReport.AddMetric(metric)
	}

	// Wrap the MetricReport in a request object.
	req := mws.NewPutMetricsForAggregationRequest()
	req.AddMetricReport(metricReport)

	if metrics.mwsClient == nil {
		// Do nothing for now...
		log.Error("MetricsClient.mwsClient implementation was null... Skipping metrics publish")
		return
	}

	_, metricsErr := metrics.mwsClient.PutMetricsForAggregation(req, mws.Regions[metricsRegion], config.GetAwsMetricsCredentials())
	if metricsErr != nil {
		log.WithError(metricsErr)
		return // Do nothing for now...but the metrics publish has failed.
	}
}
