package middleware

import (
	"net/http"

	"goji.io"

	"time"

	"code.justin.tv/samus/gateway/api/apidef"
	"code.justin.tv/samus/gateway/metrics"
	"golang.org/x/net/context"
)

// HandlerMetrics is the middleware responsible for logging latency metrics around all API
// handlers (e.g. getPrimeOffers). This happens independently of the API handler itself.
type HandlerMetrics struct {
	metrics metrics.IMetricsClient
}

// NewHandlerMetrics instantiates a new HandlerMetrics
func NewHandlerMetrics(metrics metrics.IMetricsClient) *HandlerMetrics {
	return &HandlerMetrics{
		metrics: metrics,
	}
}

// Metrics logs latency, availability, and volume metrics around all API handlers
func (m *HandlerMetrics) Metrics(h goji.Handler) goji.Handler {
	return goji.HandlerFunc(func(ctx context.Context, w http.ResponseWriter, r *http.Request) {
		loadBalancerProtocol := r.Header["X-Forwarded-Proto"]
		startTime := time.Now()
		writer := NewStatusRecorderResponseWriter(w)
		h.ServeHTTPC(ctx, writer, r)
		duration := time.Since(startTime)

		namedHandler := apidef.GetNamedHandler(ctx)
		if namedHandler != nil && m != nil && m.metrics != nil {
			handlerName := string(namedHandler.GetHandlerName())
			m.metrics.PostAvailabilityAndLatencyMetrics(handlerName, writer.statusCode, duration)
			if loadBalancerProtocol != nil {
				m.metrics.PostSecureRequestMetrics(handlerName, loadBalancerProtocol[0])
			}
		}
	})
}
