package promotion_string

import (
	"fmt"
	"strings"
	"time"

	"github.com/patrickmn/go-cache"
)

const (
	DefaultExpiration      time.Duration = 5 * time.Minute // 5 minutes
	DefaultCleanupInterval time.Duration = 0
)

type PromotionStringCacheKey struct {
	Item        string
	CountryCode string
	Locale      string
}

var ExpirationDuration time.Duration

type PromotionStringCache struct {
	cache *cache.Cache
}

type IPromotionStringCache interface {
	GetPromotionString(key PromotionStringCacheKey) (interface{}, bool)
	SetPromotionString(key PromotionStringCacheKey, value interface{})
}

func (promoCache *PromotionStringCache) GetPromotionString(key PromotionStringCacheKey) (interface{}, bool) {
	keyAsString := getPromotionStringCacheKeyAsString(key)
	promoString, found := promoCache.cache.Get(keyAsString)
	return promoString, found
}

func (promoCache *PromotionStringCache) SetPromotionString(key PromotionStringCacheKey, value interface{}) {
	keyAsString := getPromotionStringCacheKeyAsString(key)
	promoCache.cache.Set(keyAsString, value, ExpirationDuration)
}

func getPromotionStringCacheKeyAsString(key PromotionStringCacheKey) string {
	return fmt.Sprintf("%s-%s-%s", key.Item, key.CountryCode, key.Locale)
}

func getPromotionStringCacheKeyFromString(key string) PromotionStringCacheKey {
	promoCacheKeyItems := strings.Split(key, "-")
	return PromotionStringCacheKey{
		Item:        promoCacheKeyItems[0],
		CountryCode: promoCacheKeyItems[1],
		Locale:      promoCacheKeyItems[2],
	}
}

func setExpirationDuration(duration time.Duration) {
	ExpirationDuration = duration
}

func NewCache() IPromotionStringCache {
	promoCache := &PromotionStringCache{}
	setExpirationDuration(DefaultExpiration)
	cache := cache.New(ExpirationDuration, DefaultCleanupInterval)
	promoCache.cache = cache
	return promoCache
}

func NewCacheWithExpiration(duration time.Duration) IPromotionStringCache {
	promoCache := &PromotionStringCache{}
	setExpirationDuration(duration)
	cache := cache.New(duration, DefaultCleanupInterval)
	promoCache.cache = cache
	return promoCache
}
