package promotion_string

import (
	"errors"

	"strings"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbattribute"
	log "github.com/sirupsen/logrus"
)

type StringID string
type StartTime string

type GetPromotionStringResponse struct {
	ExternalUrl    string
	String         string
	IsExternalLink bool
}

type PromotionStringTable struct {
}

type PromotionString struct {
	StringID         StringID
	StartTime        StartTime
	EndTime          string
	IsDefault        string
	StringWithFilter StringWithFilter
	IsExternalLink   bool
	ExternalUrl      string
}

type StringWithFilter struct {
	Filter      Filter            `dynamodbav:"filter"`
	Strings     map[string]string `dynamodbav:"strings"`
	DefaultToEn bool              `dynamodbav:"defaultToEn"`
}

type Filter struct {
	Type string
	Data string
}

type PromotionStringDao struct {
	client *DynamoClient
}

type IPromotionStringDao interface {
	GetActiveString(stringID string, time string, countryCode string, locale string) (*GetPromotionStringResponse, error)
	GetDefaultString(stringID string, locale string) (*GetPromotionStringResponse, error)
}

func ConvertAttributeMapToPromotionString(attributeMap map[string]*dynamodb.AttributeValue) (*PromotionString, error) {
	stringWithFilter, err := ConvertStringWithFilterMapToRecord(attributeMap)
	if err != nil {
		return nil, err
	}
	return &PromotionString{
		StringID:         StringID(StringFromAttributes(attributeMap, "stringId")),
		StartTime:        StartTime(StringFromAttributes(attributeMap, "startTime")),
		ExternalUrl:      StringFromAttributes(attributeMap, "externalUrl"),
		IsExternalLink:   BoolFromAttributes(attributeMap, "isExternalLink"),
		EndTime:          StringFromAttributes(attributeMap, "endTime"),
		IsDefault:        StringFromAttributes(attributeMap, "isDefault"),
		StringWithFilter: stringWithFilter,
	}, nil
}

func ConvertStringWithFilterMapToRecord(attributeMap map[string]*dynamodb.AttributeValue) (StringWithFilter, error) {
	stringWithFilter := &StringWithFilter{}
	stringWithFilterAv, exists := attributeMap["stringWithFilter"]
	if !exists {
		log.Error("There was no string with filter object found!")
		return *stringWithFilter, nil
	}
	err := dynamodbattribute.Unmarshal(stringWithFilterAv, stringWithFilter)
	if err != nil {
		log.WithError(err).Error("Unable to unmarshal string with filter: ", err)
		return *stringWithFilter, err
	}
	return *stringWithFilter, nil
}

func NewPromotionStringDao(client *DynamoClient) IPromotionStringDao {
	dao := &PromotionStringDao{}
	dao.client = client
	return dao
}

func (dao *PromotionStringDao) GetActiveString(stringID string, time string, countryCode string, locale string) (*GetPromotionStringResponse, error) {
	tableName := dao.client.ClientConfig.TableName
	expression := "stringId = :stringId and startTime <= :startTime"

	stringIdAv := &dynamodb.AttributeValue{
		S: aws.String(stringID),
	}
	startTimeAv := &dynamodb.AttributeValue{
		S: aws.String(time),
	}

	attributeMap := make(map[string]*dynamodb.AttributeValue)
	attributeMap[":stringId"] = stringIdAv
	attributeMap[":startTime"] = startTimeAv

	queryInput := &dynamodb.QueryInput{
		TableName:                 aws.String(string(tableName)),
		KeyConditionExpression:    aws.String(expression),
		ExpressionAttributeValues: attributeMap,
	}

	result, err := dao.client.Dynamo.Query(queryInput)
	if err != nil {
		return nil, err
	}
	for _, item := range result.Items {
		promoString, err := ConvertAttributeMapToPromotionString(item)
		if err != nil {
			log.WithError(err).Error("Failed to query for string", err)
			return nil, err
		}
		if time < promoString.EndTime {
			stringWithFilter := promoString.StringWithFilter
			if stringWithFilter.Filter.isActive(countryCode) {
				stringMap := stringWithFilter.Strings
				//first check for a translated string
				if val, ok := stringMap[strings.ToLower(locale)]; ok {
					response := GetPromotionStringResponse{
						ExternalUrl:    promoString.ExternalUrl,
						String:         val,
						IsExternalLink: promoString.IsExternalLink,
					}
					log.Info("Found active string for locale: ", strings.ToLower(locale), ", for string: ", stringID)
					return &response, nil
				}
				//then default if required
				if stringWithFilter.DefaultToEn {
					if val, ok := stringMap["en"]; ok {
						response := GetPromotionStringResponse{
							ExternalUrl:    promoString.ExternalUrl,
							String:         val,
							IsExternalLink: promoString.IsExternalLink,
						}
						log.Info("Defaulting to En for active string: ", stringID)
						return &response, nil
					} else {
						return nil, errors.New("Unable to find defaultToEn string for: " + stringID)
					}
				}
			}
		}
	}
	return nil, nil
}

// Runs the ruleset of the 'type' of filter. Current supported Types are:
// Whitelist: Only listed countries see the string
// Blacklist: Only listed countries do *not* see the string
// All: All countries see the string
func (filter *Filter) isActive(countryCode string) bool {
	if filter.Type == "ALL" {
		return true
	}
	if filter.Type == "WHITELIST" {
		countries := strings.Split(filter.Data, ",")
		return contains(countries, countryCode)
	}
	if filter.Type == "BLACKLIST" {
		countries := strings.Split(filter.Data, ",")
		return !contains(countries, countryCode)
	}
	return false
}

func contains(s []string, e string) bool {
	for _, a := range s {
		if strings.ToLower(a) == strings.ToLower(e) {
			return true
		}
	}
	return false
}

func (dao *PromotionStringDao) GetDefaultString(stringID string, locale string) (*GetPromotionStringResponse, error) {
	tableName := dao.client.ClientConfig.TableName
	expression := "stringId = :stringId and isDefault = :isDefault"

	stringIdAv := &dynamodb.AttributeValue{
		S: aws.String(stringID),
	}
	defaultAv := &dynamodb.AttributeValue{
		S: aws.String("true"),
	}

	attributeMap := make(map[string]*dynamodb.AttributeValue)
	attributeMap[":stringId"] = stringIdAv
	attributeMap[":isDefault"] = defaultAv

	queryInput := &dynamodb.QueryInput{
		TableName:                 aws.String(string(tableName)),
		KeyConditionExpression:    aws.String(expression),
		ExpressionAttributeValues: attributeMap,
		IndexName:                 aws.String("stringId-isDefault-index"),
	}

	result, err := dao.client.Dynamo.Query(queryInput)
	if err != nil {
		log.WithError(err).Error("Failed to query for default string", err)
		return nil, err
	}
	for _, item := range result.Items {
		promoString, err := ConvertAttributeMapToPromotionString(item)
		if err != nil {
			log.WithError(err).Error("Failed to convert string", err)
			return nil, err
		}
		stringWithFilter := promoString.StringWithFilter
		stringMap := stringWithFilter.Strings
		//first check for a translated string
		if val, ok := stringMap[strings.ToLower(locale)]; ok {
			response := GetPromotionStringResponse{
				ExternalUrl:    promoString.ExternalUrl,
				String:         val,
				IsExternalLink: promoString.IsExternalLink,
			}
			log.Info("Found default string for locale: ", strings.ToLower(locale), ", for string: ", stringID)
			return &response, nil
		}
		//then default if required
		if stringWithFilter.DefaultToEn {
			val, _ := stringMap["en"]
			response := GetPromotionStringResponse{
				ExternalUrl:    promoString.ExternalUrl,
				String:         val,
				IsExternalLink: promoString.IsExternalLink,
			}
			log.Info("Defaulting to En for default string: ", stringID)
			return &response, nil
		}
	}
	log.Warn("Failed to find default string for: ", stringID)
	return nil, errors.New("Unable to find default string for: " + stringID)
}
