package sns

import (
	"fmt"

	log "github.com/sirupsen/logrus"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/credentials"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/sns"
)

type SnsClientConfig struct {
	AwsRegion           string
	TopicArn            string
	EndpointOverride    string
	CredentialsOverride *credentials.Credentials
}

type SnsClient struct {
	Sns          *sns.SNS
	ClientConfig *SnsClientConfig
}

type SnsPublisher interface {
	Publish(message string) error
}

// Creates a client from an sns client
func NewClientFromSNS(sns *sns.SNS) SnsPublisher {
	snsClient := &SnsClient{
		Sns: sns,
	}
	return snsClient
}

// Creates a new SNS Client
func NewClient(config *SnsClientConfig) SnsPublisher {
	awsConfig := &aws.Config{
		Region:      aws.String(config.AwsRegion),
		Credentials: config.CredentialsOverride,
	}
	snsClient := sns.New(session.New(), awsConfig)

	return &SnsClient{
		Sns:          snsClient,
		ClientConfig: config,
	}
}

// Publish an SNS Event
func (snsClient *SnsClient) Publish(message string) error {
	msg := fmt.Sprintf("SnsClient: Publishing SNS message, %v", message)
	log.Info(msg)

	params := &sns.PublishInput{
		Message:   aws.String(message),
		TargetArn: aws.String(snsClient.ClientConfig.TopicArn),
	}

	_, err := snsClient.Sns.Publish(params)
	if err != nil {
		errorMsg := fmt.Sprintf("SnsClient: Failed to Publish event, %v", err)
		log.WithError(err).Error(errorMsg)
		return err
	}
	return nil
}
