package util

import (
	. "github.com/smartystreets/goconvey/convey"
	"testing"
)

func TestValidateOffer(t *testing.T) {
	Convey("ValidateOffer", t, func() {
		Convey("Returns an error if offerId is of invalid length", func() {
			offerId := "2ed07990-971-4aa4-a30e-b5f628cfd707"
			So(ValidateOffer(offerId), ShouldNotBeNil)
		})

		Convey("Returns an error if offerId contains unexpected characters", func() {
			offerId := "2ed07990-9d71-4aa4-a3*e-b5f628cfd707"
			So(ValidateOffer(offerId), ShouldNotBeNil)
		})

		/**
		 * UUID uses hexadecimal, therefore any letter past 'F' will be deemed invalid
		 */
		Convey("Returns an error if offerId contains invalid alphanumeric characters", func() {
			offerId := "2ed07990-9d71-4aa4-a30g-b5f628cfd707"
			So(ValidateOffer(offerId), ShouldNotBeNil)
		})

		Convey("Returns an error of offerId contains incorrect number of tuples", func() {
			offerId := "2ed07990-9d71-4aa4-b5f628cfd707"
			So(ValidateOffer(offerId), ShouldNotBeNil)
		})

		Convey("Returns an error if offerId is empty", func() {
			offerId := ""
			So(ValidateOffer(offerId), ShouldNotBeNil)
		})

		Convey("Returns no error with valid offerId", func() {
			offerId := "2ab52aef-0428-7aa2-f043-d28ce0544fd9"
			So(ValidateOffer(offerId), ShouldBeNil)
		})
	})
}

func TestValidateOffers(t *testing.T) {
	Convey("ValidateOffers", t, func() {
		Convey("Returns an error if any one offerId is of invalid length", func() {
			offerIds := []string{
				"2ab52aef-0428-7aa2-f043-d28ce0544fd9",
				"2ed07990-971-4aa4-a30e-b5f628cfd707",
			}
			So(ValidateOffers(offerIds...), ShouldNotBeNil)
		})

		Convey("Returns an error if any one offerId contains unexpected characters", func() {
			offerIds := []string{
				"2ab52aef-0428-7aa2-f043-d28ce0544fd9",
				"2ed07990-9d71-4aa4-a3*e-b5f628cfd707",
			}
			So(ValidateOffers(offerIds...), ShouldNotBeNil)
		})

		Convey("Returns an error if any one offerId contains invalid alphanumeric characters", func() {
			offerIds := []string{
				"2ab52aef-0428-7aa2-f043-d28ce0544fd9",
				"2ed07990-9d71-4aa4-a30g-b5f628cfd707",
			}
			So(ValidateOffers(offerIds...), ShouldNotBeNil)
		})

		Convey("Returns an error if any one offerId contains incorrect number of tuples", func() {
			offerIds := []string{
				"2ab52aef-0428-7aa2-f043-d28ce0544fd9",
				"2ed07990-9d71-4aa4-b5f628cfd707",
			}
			So(ValidateOffers(offerIds...), ShouldNotBeNil)
		})

		Convey("Returns an error if any one offerId is empty", func() {
			offerIds := []string{
				"2ab52aef-0428-7aa2-f043-d28ce0544fd9",
				"",
			}
			So(ValidateOffers(offerIds...), ShouldNotBeNil)
		})

		Convey("Returns no error if all offerIds are valid", func() {
			offerIds := []string{
				"2ab52aef-0428-7aa2-f043-d28ce0544fd9",
				"54b5ca60-9a1a-0f53-7f47-ee2157893bc2",
				"84b44d9d-8153-75f2-00d9-46bfaa309d48",
				"2cb4cf2c-cd2f-c5e1-bd13-ebb2d1032f2e",
				"20b56677-7f46-4e70-a70e-29361ca72c17",
				"18b511b3-19ed-337e-a635-f13f57420360",
			}
			So(ValidateOffers(offerIds...), ShouldBeNil)
		})

		Convey("Returns no error with past Beta offerIds", func() {
			So(ValidateOffers(BetaOfferIds...), ShouldBeNil)
		})

		Convey("Returns no error with past Prod offerIds", func() {
			So(ValidateOffers(ProdOfferIds...), ShouldBeNil)
		})
	})
}
