package util

import (
	. "github.com/smartystreets/goconvey/convey"
	"testing"
)

func TestValidateTuid(t *testing.T) {
	Convey("ValidateTuid", t, func() {
		Convey("Returns an error if input is not alphanumeric", func() {
			tuid := "1235*"
			So(ValidateTuid(tuid), ShouldNotBeNil)
		})

		Convey("Returns an error if input is not numeric", func() {
			tuid := "WeNeedMoreUnitTests"
			So(ValidateTuid(tuid), ShouldNotBeNil)
		})

		Convey("Returns an error if input is a float", func() {
			tuid := "3.14159265359"
			So(ValidateTuid(tuid), ShouldNotBeNil)
		})

		Convey("Returns an error if input is a zero", func() {
			tuid := "0"
			So(ValidateTuid(tuid), ShouldNotBeNil)
		})

		Convey("Returns an error if input is negative", func() {
			tuid := "-314159265359"
			So(ValidateTuid(tuid), ShouldNotBeNil)
		})

		Convey("Returns no error if input is numeric", func() {
			tuid := "314159265359"
			So(ValidateTuid(tuid), ShouldBeNil)
		})

		Convey("Returns no error at extrema", func() {
			tuid := "9223372036854775807"
			So(ValidateTuid(tuid), ShouldBeNil)
		})
	})
}

func TestValidateTuids(t *testing.T) {
	Convey("ValidateTuids", t, func() {
		Convey("Returns an error if any one TUID is not alphanumeric", func() {
			tuids := []string{
				"314159265359",
				"314159265359*",
			}
			So(ValidateTuids(tuids...), ShouldNotBeNil)
		})

		Convey("Returns an error if any one TUID is not numeric", func() {
			tuids := []string{
				"314159265359",
				"WeNeedMoreUnitTests",
			}
			So(ValidateTuids(tuids...), ShouldNotBeNil)
		})

		Convey("Returns an error if any one TUID is a float", func() {
			tuids := []string{
				"314159265359",
				"3.14159265359",
			}
			So(ValidateTuids(tuids...), ShouldNotBeNil)
		})

		Convey("Returns an error if any one TUID is negative", func() {
			tuids := []string{
				"314159265359",
				"-314159265359",
			}
			So(ValidateTuids(tuids...), ShouldNotBeNil)
		})

		Convey("Returns no error if all TUIDs are numeric", func() {
			tuids := []string{
				"314159265359",
				"628318530717",
			}
			So(ValidateTuids(tuids...), ShouldBeNil)
		})

		Convey("Returns no error at extrema", func() {
			tuids := []string{
				"9223372036854775807",
				"1",
			}
			So(ValidateTuids(tuids...), ShouldBeNil)
		})
	})
}
