package workers

import (
	log "github.com/sirupsen/logrus"

	"github.com/cactus/go-statsd-client/statsd"

	"code.justin.tv/chat/workerqueue"
	"code.justin.tv/common/config"

	"code.justin.tv/samus/gateway/backend"
	"fmt"
)

var (
	AWSAccessKeyID     = "aws_access_key_id"
	AWSSecretAccessKey = "aws_secret_access_key"
	AWSRegionName      = "aws-region-name"
)

func init() {
	config.Register(map[string]string{
		AWSAccessKeyID:     "",
		AWSSecretAccessKey: "",
		AWSRegionName:      "us-west-2",
	})
}

// InitSQSWorkers initializes workers to process updates from Samus
func InitSQSWorkersForSamus(statsdClient statsd.Statter, backend *backend.Backend, numWorkers int, queueName string) {
	doneCh := make(chan struct{}, numWorkers)
	workerParams := workerqueue.CreateWorkersParams{
		QueueName: queueName,
		Region:    "us-west-2",
		Tasks: map[workerqueue.TaskVersion]workerqueue.TaskFn{
			-1: backend.ProcessSamusSQSNotification,
		},
		NumWorkers: numWorkers,
	}
	wg, errCh, err := workerqueue.CreateWorkers(workerParams, doneCh, statsdClient)

	for err := range errCh {
		log.Debugf("Message failed to process with err: %s", err)
	}

	if err != nil {
		log.WithError(err).Error(fmt.Errorf("error starting up workers: %s", err))
	}

	wg.Wait()
}

// InitSQSWorkersForOrders initializes workers to process updates from SamusOrderingService (SORDS)
func InitSQSWorkersForOrders(statsdClient statsd.Statter, backend *backend.Backend, numWorkers int, queueName string) {
	doneCh := make(chan struct{}, numWorkers)
	workerParams := workerqueue.CreateWorkersParams{
		QueueName: queueName,
		Region:    "us-west-2",
		Tasks: map[workerqueue.TaskVersion]workerqueue.TaskFn{
			-1: backend.ProcessOrderSQSNotification,
		},
		NumWorkers: numWorkers,
	}
	wg, errCh, err := workerqueue.CreateWorkers(workerParams, doneCh, statsdClient)

	for err := range errCh {
		log.Debugf("Message failed to create workers for Orders SQS with err: %s", err)
	}

	if err != nil {
		log.WithError(err).Error(fmt.Errorf("error starting up Order SQS workers: %s", err))
	}

	wg.Wait()
}
