// Copyright 2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.

const serverIP = require("../config/endpoints.js").serverIP
const GameCoinflipClient = require("./GameCoinflipClient.js")
const GameTimer = require("../../common/GameTimer.js")
var io = require('socket.io-client');
var socket = io.connect(serverIP+":8000")

const CONNECTION_STATUS = {
    CONNECTED: "CONNECTED",
    DISCONNECTED: "DISCONNECTED",
    RECONNECTING: "RECONNECTING"
}


class GameManagerClient {

    constructor() {
        this.setupListeners()

        this.playerName = null
        this.playersInGame = []
        this.isAdmin = false
        this.playerID = null
        this.connectionStatus = CONNECTION_STATUS.DISCONNECTED
        this.spectating = false

        this.gameTimer = new GameTimer.GameTimer(this.startGame.bind(this))

        this.gameClient = null

        this.lastCorrectAnswer = null
        this.lostOnTurn = -1
        this.turn = 0

        this.sessionOver = false;

        // necessary binds for invocation from the views
        this.sendStartSessionMessage = this.sendStartSessionMessage.bind(this)
    }

    update(delta) {
        this.gameTimer.update()
        if(this.gameClient !== undefined && this.gameClient !== null) {
            if (this.gameClient.update instanceof Function) {
                this.gameClient.update(delta)
            }
        }
    }

    setupListeners() {

        socket.on('welcome', (data) => {
            this.isAdmin = data.isAdmin
            this.playerID = data.playerID
            this.updatePlayersInGame(data.connectedPlayers)
        });

        socket.on('lateWelcome', (data) => {
            this.isAdmin = data.isAdmin
            this.playerID = data.playerID
            this.updatePlayersInGame(data.connectedPlayers)

            switch (data.currentState) {
                case GameTimer.STATE.PLAYING:
                    this.gameTimer.startPlayingSince(data.currentRoundStartTime)
                    break
                case GameTimer.STATE.RESULTS:
                    this.gameTimer.startResultsSince(data.currentRoundStartTime)
                    break
                default:
                    this.gameTimer.state = data.currentState
                    this.gameTimer.stateStart = data.currentRoundStartTime
            }
            this.spectating = true
        });

        socket.on('sessionStart', (data) => {
            if (this.playerName != null) {
                this.gameTimer.startPlayingSince(data.startTime)
                this.updatePlayersInGame(data.playersLeftInGame)
                this.spectating = false
            }
        });

        socket.on('playersInGame', (data) => {
            this.updatePlayersInGame(data.playersLeftInGame)
        });

        socket.on('gameEnd', (data) => {
            // stop playing if timer hasn't stopped already
            if (this.gameTimer.state == GameTimer.STATE.PLAYING) {
                this.gameTimer.startResultsSince(data.endTime)
            }
            this.lastCorrectAnswer = data.correctAnswer
            this.updatePlayersInGame(data.playersLeftInGame)
            this.gameClient.won = this.stillInTheGame
            this.gameClient.allPlayersLost = data.allPlayersLost
            if (!this.stillInTheGame && this.lostOnTurn < 0) {
                this.lostOnTurn = this.turn
            }

            this.sessionOver = data.sessionOver
            if (this.sessionOver) {
                this.gameTimer.end()
            }
        });

        socket.on('newAdmin', (data) => {
            this.isAdmin = true
        });

        socket.on('connect', () => {
            this.connectionStatus = CONNECTION_STATUS.CONNECTED
        })
        socket.on('disconnect', () => {
            this.connectionStatus = CONNECTION_STATUS.DISCONNECTED.DISCONNECTED
        })
        socket.on('reconnecting', () => {
            this.connectionStatus = CONNECTION_STATUS.DISCONNECTED.RECONNECTING
        })
        socket.on('reconnect', () => {
            this.connectionStatus = CONNECTION_STATUS.DISCONNECTED.CONNECTED
        })
    }

    connectToServer(playerName) {
        this.playerName = playerName
        socket.emit('joinSession', playerName)
    }

    startGame() {
        this.turn++
        this.gameClient = new GameCoinflipClient.GameCoinflipClient(this)
    }

    updatePlayersInGame(playersInGame) {
        this.playersInGame = playersInGame
    }

    // Only an admin player should be able to run this
    sendStartSessionMessage() {
        socket.emit('startSession')
    }

    sendPlayerInput(input) {
        socket.emit('playerInput', input)
    }

    get gameState() {
        return this.gameTimer.state
    }

    get standbyForResult() {
        return this.gameClient.standbyForResult
    }

    get standbyForResultSummary() {
        return this.gameClient.standbyForResultSummary
    }

    get resultSummary() {
        return this.gameClient.resultSummary
    }

    get stillInTheGame() {
        let playerIDs = this.playersInGame.map(player => player.playerID)
        return playerIDs.includes(this.playerID)
    }

    get timerIsActive() {
        return this.gameTimer.timerIsActive
    }

    get getProgressToNextPhase() {
        return this.gameTimer.getProgressToNextPhase
    }

}


module.exports = {GameManagerClient, CONNECTION_STATUS}
