// Copyright 2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.

const React = require("react")

const model = require("../models/GameManagerClient.js")

const GameCoinflip = require("./GameCoinflip.view.js")
const GameTimer = require("../../common/GameTimer.js")
const PlayerList = require("./PlayerList.js")

require("../views/View.less")

module.exports = class View extends React.Component {
    constructor(props) {
        super(props)
        this.model = props.clientModel

        this.state = {playerName: '', invalidName: false};

        this.handleNameChange = this.handleNameChange.bind(this);
        this.handleNameSubmit = this.handleNameSubmit.bind(this);
    }

    render() {
        return (
            <div className="ExampleView">
                {this.renderProgressBar()}
                <div className="CoinflipGame">
                    <div className="game-contents">
                        {this.renderGameManager()}
                    </div>
                </div>
            </div>
        )
    }

    handleNameChange(event) {
        this.setState({invalidName: false})
        this.setState({playerName: event.target.value})
    }
    handleNameSubmit(event) {
        this.setState({invalidName: this.state.playerName == ''}, () => {
            if (!this.state.invalidName) {
                this.props.clientModel.connectToServer(this.state.playerName)
            }
        })
        event.preventDefault();
    }

    renderGameManager() {
        let timeLeft = this.model.gameTimer.secondsLeftInState

        if (this.model.connectionStatus != model.CONNECTION_STATUS.CONNECTED) {
            return (
                <div className='title'>Not connected to server</div>
            )
        }

        switch (this.model.gameState) {
            case GameTimer.STATE.WAITING:
                if (this.model.playerName == null) {
                    return (
                        <div>
                            <form onSubmit={this.handleNameSubmit}>
                                <label>
                                    Name:
                                    <input type="text" value={this.state.playerName} onChange={this.handleNameChange} />
                                </label>
                                <input type="submit" value="Join Game" />
                            </form>
                            <p className={'lose '+(this.state.invalidName? '': 'hide')}>Invalid Name</p>
                        </div>
                    )
                }
                if (this.model.playersInGame == undefined || this.model.playersInGame.length == 0) {
                    return (
                        <div className='title'>Error connecting to the server</div>
                    )
                }
                return (
                    <div>
                        <div className='game-title'>The Choice Game</div>
                        <div className='prime-upsell'><h3>Presented by <span className='light-purple'>Twitch Prime</span></h3></div>
                        <div className='subtext'>Waiting for the admin to start the game</div>
                        <PlayerList players={this.model.playersInGame}/>
                        <button 
                            className={'button-start-game' + (this.model.isAdmin? '' : ' hide')}
                            onMouseDown={this.model.sendStartSessionMessage}
                        >
                            Start Game
                        </button>
                    </div>
                )
            case GameTimer.STATE.PLAYING:

                if (!this.model.stillInTheGame) {
                    return (
                        <div>
                            <div className='light-purple title'>
                                {this.model.spectating? 'Spectating ongoing game' : 'You already lost...'}
                            </div>
                            <div>...but the game goes on!</div>
                            <div className='subtext'>{'The coin will be flipped in ' + timeLeft + 's'}</div>
                        </div>
                    )
                }
                if (this.model.standbyForResult) {
                    return (
                        <div>
                            <div className='title'>{this.model.standbyForResultSummary}</div>
                            <div className='subtext'>{'The coin will be flipped in ' + timeLeft + 's'}</div>
                        </div>
                    )
                }
                return (
                    <GameCoinflip clientModel = {this.props.clientModel}/>
                )

            case GameTimer.STATE.RESULTS:
                let titleClassResults = ''
                if (this.model.stillInTheGame) {
                    // Answered Right
                    titleClassResults = 'win'
                } else if (this.model.lostOnTurn == this.model.turn && !this.model.spectating) {
                    // Answered Wrong or didn't answer
                    titleClassResults = 'lose'
                } else {
                    // Lost previously, or just joined the game (spectating)
                    titleClassResults = 'light-purple'
                }
                let messageResults = "Spectating ongoing game"
                if (!this.model.spectating) {
                    messageResults = this.model.resultSummary
                }
                return (
                    <div>
                        <div className={'title '+titleClassResults}>{messageResults}</div>
                        <div>The coin landed on {this.model.lastCorrectAnswer}</div>
                        <div className='subtext'>Players left:</div>
                        <PlayerList players={this.model.playersInGame}/>
                        <div>{'Next round in ' + timeLeft + 's'}</div>
                    </div>
                )
            case GameTimer.STATE.COMPLETE:
                  // You Wonw
                if (this.model.stillInTheGame) {
                    return (
                        <div>
                            <div className='win title'>Winner!</div>
                            <div>You're the last player standing</div>
                            {this.renderRestartButton()}
                        </div>
                    )
                } 
                // All lost (should not longer happen)
                if (this.model.playersInGame.length == 0) {
                    return (
                        <div>
                            <div className='lose title'>No Winner!</div>
                            <div>All remaining players lost in the last round</div>
                            {this.renderRestartButton()}
                        </div>
                    )
                }
                // You were spectating
                if (this.model.spectating) {
                    return (
                        <div>
                            <div className='light-purple title'>The game ended</div>
                            <div>{this.model.playersInGame[0].playerName} won the game</div>
                            {this.renderRestartButton()}
                        </div>
                    )
                }
                // You lost
                return (
                    <div>
                        <div className='lose title'>You lost!</div>
                        <div>{this.model.playersInGame[0].playerName} won the game</div>
                            {this.renderRestartButton()}
                    </div>
                )
            default:
                return <div>Sorry, something went wrong</div>

        }
    }

    renderRestartButton() {

        if (this.model.isAdmin) {
            return (
                <button 
                    className={'button-start-game' + (this.model.isAdmin? '' : ' hide')}
                    onMouseDown={this.model.sendStartSessionMessage}
                >
                    Start Game
                </button>
            )
        }
        return (
            <div className='subtext'>Waiting for the admin restart the game</div>
        )
    }

    renderProgressBar() {
        let model = this.props.clientModel
        if (!model.timerIsActive) {
            return;
        }

        let progress = model.getProgressToNextPhase * 100;
        
        return (
            <div className="progress-bar">
                <div className="progress-bar-fill" style={{width: progress+'%'}}></div>
            </div>
        )
    }
}
