// Copyright 2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.

const GameTimer = require("../common/GameTimer.js")
const GameCoinflipServer = require("./GameCoinflipServer.js")
var io = require('socket.io');
var http = require('http');
var server = http.createServer();
const port = 8000;
console.log('\n\n-----------------------\nlistening on port '+port+'\n-----------------------\n\n');
server.listen(port, '10.53.178.150');
var serverSocket = io.listen(server);

class GameManagerServer {

    constructor() {
        this.startServer()
        this.gameTimer = new GameTimer.GameTimer(this.startGame.bind(this), this.endGame.bind(this))

        this.connectedPlayers = [] // {playerID, playerName}
        this.playersLeftInGame = [] // {playerID, playerName}
        this.adminID = null;
    }

    startServer() {
        serverSocket.on('connection', (client) => {
            console.log ("A new user connected ("+client.id+")");

            client.on('joinSession', (playerName) => {
                this.connectPlayer(client, playerName)
                console.log(client.id + " set their name to  '"+playerName)
            })

            client.on('startSession', () => {
                this.startSession(client)
            })

            client.on('playerInput', (input) => {
                this.processPlayerInput(client, input)
            })
            
            client.on('disconnect', (reason) => {
                console.log("A player disconnected ("+client.id+")")
                this.removePlayer(client)
            })
        });
    }

    getConnectedUserIDs() {
        return Object.keys(serverSocket.sockets.sockets)
    }

    printUserList() {
        let clientIDList = this.getConnectedUserIDs()
        let players = []
        clientIDList.forEach((clientID) => {
            let playerObj = this.findClientByID(clientID)
            if (playerObj.playerName != null && playerObj.playerID != null) {
                players.push("- "+playerObj.playerName +" ("+ playerObj.playerID +")")
            } else {
                players.push("- Unregistered player ("+ clientID+")")
            }
        })
        console.log(clientIDList.length + " client(s) connected, " + players.length + " with name(s):")
        console.log(players.join('\n'))
        console.log("\n")
    }

    findClientByID(id) {
        let matchingClients = this.connectedPlayers.filter(player => id == player.playerID)
        if (matchingClients.length == 0) {
            return {}
        } else if (matchingClients.length > 1) {
            console.log("WARNING: more than one client with id: "+id+" was found")
        }
        return matchingClients[0]
    }

    update(delta) {
    }

    removePlayer(client) {
        this.connectedPlayers = this.connectedPlayers.filter(player => client.id != player.playerID)
        if (this.state != GameTimer.STATE.COMPLETE) {
            this.playersLeftInGame = this.playersLeftInGame.filter(player => client.id != player.playerID)
        }
        
        // TODO: define new admin/handle admin disconnect
        if (client.id == this.adminID) {
            if (this.connectedPlayers.length > 0) {
                let newAdminID = this.connectedPlayers[0].playerID
                serverSocket.to(newAdminID).emit('newAdmin')
                this.setAdmin(newAdminID)
            } else {
                this.adminID = null
            }
        }

        this.sendPlayersLeftInGame()
    }

    setAdmin(newAdminID) {
        this.adminID = newAdminID
    }

    startGame() {
        this.gameServer = new GameCoinflipServer(this.playersLeftInGame)
    }

    endGame() {
        let allPlayersLost = this.gameServer.winningPlayers.length <= 0
        if (!allPlayersLost) {
            this.playersLeftInGame = this.playersLeftInGame.filter(player => this.gameServer.winningPlayers.includes(player.playerID))
        }
        // send results to client
        this.sendGameEnd(allPlayersLost)
    }

    get timeLeftInGame() {
        return TIME_PER_GAME - (Date.now() - this.gameStartTime)
    }

    connectPlayer(client, playerName) {

        // first player is admin
        if (this.adminID == null) {
            this.setAdmin(client.id)
        }

        let playerObj = {
            playerID: client.id,
            playerName: playerName, // TODO: let clients set their names/use their twitch name
        }
        this.connectedPlayers.push(playerObj)

        // prevent joining ongoing session (set in spectate mode)
        if (this.state != GameTimer.STATE.WAITING) {
            // send the start game (not session) message to all clients
            client.emit('lateWelcome', {
                currentRoundStartTime: this.gameTimer.stateStart,
                currentState: this.state,
                connectedPlayers: this.playersLeftInGame,
                isAdmin: this.adminID == client.id,
                playerID: client.id
            })
            return
        }

        // send welcome packet back to client
        client.emit('welcome', {
            connectedPlayers: this.connectedPlayers,
            isAdmin: this.adminID == client.id,
            playerID: client.id
        })
        
        // send player list to everyone
        this.playersLeftInGame.push(playerObj)
        this.sendPlayersLeftInGame()
    }

    getPlayersLeftNames() {
        let playersLeftNames = this.playersLeftInGame.map(player => player.playerName)
        return playersLeftNames
    }

    sendPacketToAllPlayers(eventName, packet) {
        serverSocket.emit(eventName, packet)
    }

    startSession (client) {
        if (this.adminID != client.id) {
            console.log("Non admin player tried to start a session")
            return
        }

        this.playersLeftInGame = [...this.connectedPlayers]
        console.log("\nStarting session.")
        this.printUserList()

        this.gameTimer.startPlayingNow()

        // send the start game (not session) message to all clients
        this.sendPacketToAllPlayers('sessionStart', {
            startTime: this.gameTimer.stateStart,
            playersLeftInGame: this.playersLeftInGame
        })
    }

    sendGameEnd (allPlayersLost) {
        console.log("Game ended. Correct choice was "+this.gameServer.correctChoice);

        let sessionOver = this.playersLeftInGame.length <= 1
        if (sessionOver) {
            console.log("Session ended")
            this.gameTimer.end()
        }

        this.sendPacketToAllPlayers('gameEnd', {
            playersLeftInGame: this.playersLeftInGame,
            endTime: this.gameTimer.stateStart,
            correctAnswer: this.gameServer.correctChoice,
            allPlayersLost: allPlayersLost,
            sessionOver: sessionOver
        })
    }

    sendPlayersLeftInGame() {
        this.sendPacketToAllPlayers('playersInGame', {
            playersLeftInGame: this.playersLeftInGame
        })
    }

    processPlayerInput(client, input) {
        if (this.isStillInTheGame(client.id)) {
            this.gameServer.processPlayerInput(client.id, input)
        } else {
            // TODO: test this
            console.log("A PLAYER WHO WAS ELIMINATED TRIED TO PLAY")
        }
    }

    isStillInTheGame(playerID) {
        let result = false
        this.playersLeftInGame.forEach((player) => {
            if (player.playerID == playerID) {
                result = true
            }
        })
        return result
    }

    get state() {
        return this.gameTimer.state
    }
}


module.exports = new GameManagerServer() // a singleton
