#!/usr/bin/env powershell
#Requires -Version 3.0


#######################################
#
# Parameters
#
#######################################
param(
    [switch]$PauseOnExit    # Prevent administrative shell from exiting
)


#######################################
#
# Functions
#
#######################################
function LogDebug ()
{
    Write-Host -foregroundcolor cyan $args
}

function LogInfo ()
{
    Write-Host -foregroundcolor white $args
}

function LogWarn ()
{
    Write-Host -foregroundcolor yellow "WARN: $args"
}

function LogError ()
{
    Write-Host -foregroundcolor red "ERROR: $args"
}

function Fatal ()
{
    Write-Host -foregroundcolor red "ERROR: $args"
    if ($PauseOnExit) { pause }
    exit 1
}

function ReinvokeScriptAsAdmin ()
{
    # Elevate to gain admin rights if necessary
    $isAdmin = ([Security.Principal.WindowsPrincipal] [Security.Principal.WindowsIdentity]::GetCurrent()).IsInRole(
        [Security.Principal.WindowsBuiltInRole]::Administrator
    );
    if (!$isAdmin)
    {
        LogInfo "Elevating script to gain administrative rights..."
        $CommandLine = "-File `"$($MyInvocation.ScriptName)`" -PauseOnExit "
        try
        {
            Start-Process -FilePath PowerShell.exe -Wait -Verb Runas -ArgumentList $CommandLine
        }
        catch
        {
            LogError $_.Exception.Message
        }
        if ($PauseOnExit) { pause }
        exit
    }
}

function InstallChocolatey ()
{
    if (-not (Get-Command "chocolatey.exe" -ErrorAction SilentlyContinue))
    {
        # Set-ExecutionPolicy Bypass -Scope Process -Force;
        iex ((New-Object System.Net.WebClient).DownloadString('https://chocolatey.org/install.ps1'))
    }
}


#######################################
#
# Main
#
#######################################
try
{
    ReinvokeScriptAsAdmin
    InstallChocolatey

    # Node is used for the game server
    choco install nodejs --version 10.12.0 --confirm

    # Shellcheck used to validate bash scripts
    choco install shellcheck --confirm

    # Java Runtime is used to run GameLift-Local
    choco install jre8 --confirm

    # jq is used to extract data from AWSCLI for bin/ssh-instance
    choco install jq --confirm
}
catch
{
    Fatal $_.Exception.Message
}

if ($PauseOnExit) { pause }
