'use strict';

module.exports = class RateLimiter
{
    /*
     * class constructor
     *
     * @param <integer> rate limiter allowance in milliseconds
     */
    constructor(allowanceMs)
    {
        this.throttleTime = Date.now();
        this.allowanceMs = allowanceMs;
    }

    /*
     * throttle a function
     *
     * @param <integer> cost of the operation in milliseconds
     */
    throttle(costMs)
    {
        // If cost greater than allowance then the transaction would never be allowed!
        if (costMs > this.allowanceMs)
        {
            throw "Invalid cost value is greater than allowance";
        }

        // If the throttle time is in the past then set it to the current time
        const currTime = Date.now();
        if (this.throttleTime < currTime)
        {
            this.throttleTime = currTime;
        }

        // Has the caller accrued too much time-cost?
        const throttleMs = this.throttleTime + costMs;
        const allowedMs = currTime + this.allowanceMs;
        if (throttleMs > allowedMs)
        {
            return false;
        }

        this.throttleTime = throttleMs;
        return true;
    }

};
