/* global describe it */
const assert = require('assert');
const MockDate = require('mockdate');
const RateLimiter = require('../lib/rate-limiter');

describe('RateLimiter', function()
{
    // Ensure there are no numerical precision errors for the next couple thousand years
    const date1 = Date.UTC(96, 1, 2, 3, 4, 5);
    const date2 = Date.UTC(4000, 1, 2, 3, 4, 5);

    function checkDate(date, string)
    {
        MockDate.set(date);
        assert.equal(new Date().toUTCString(), string);
        const now = Date.now();
        assert.equal(now, date);
    }

    function checkRateLimitDate(allowance, increment, date)
    {
        MockDate.set(date);
        var limiter = new RateLimiter(allowance);
        for (var i = 0; i < allowance; i += increment)
        {
            assert(limiter.throttle(increment));
        }
        assert(!limiter.throttle(1));
    }

    function checkRateLimit (allowance, increment)
    {
        checkRateLimitDate(allowance, increment, date1);
        checkRateLimitDate(allowance, increment, date2);
    }

    function checkRateLimitReset(date)
    {
        MockDate.set(date);
        var limiter = new RateLimiter(1000);
        assert(limiter.throttle(500));
        assert(limiter.throttle(500));
        assert(!limiter.throttle(1));

        // Advance time to reset half of allowance
        MockDate.set(new Date(date + 500));

        // Use the other half of the allowance
        for (var i = 0; i < 500; ++i)
        {
            assert(limiter.throttle(1));
        }

        // No more time
        assert(!limiter.throttle(1));
    }

    it('MockDate should set the date correctly', function()
    {
        checkDate(date1, "Fri, 02 Feb 1996 03:04:05 GMT");
        checkDate(date2, "Wed, 02 Feb 4000 03:04:05 GMT");
    });

    it('Rate limiter should throttle when time expires', function()
    {
        checkRateLimit(1000, 1);
        checkRateLimit(1000, 250);
    });

    it('Rate limiter should reset when it falls behind', function()
    {
        checkRateLimitReset(date1);
        checkRateLimitReset(date2);
    });
});
