#!/usr/bin/env node

const gameServerModule = require('./index.js');
let gameServerRunner = new gameServerModule.GameServerRunner();

function Stringize(object) {
    let result = "";
    for (let prop in object) {
        if (object.hasOwnProperty(prop)) {
            result = result + ";" + prop + "=" + object[prop];
        }
    }
    return result;
}

// Import GameLift
const GameLift = require('@dplusic/gamelift-nodejs-serversdk');
const GameLiftServerAPI = GameLift.default;

// On program shutdown notify GameLift
const exitHook = require('exit-hook');
exitHook(() => {
    GameLiftServerAPI.Destroy();
});

// InitSDK will establish a local connection with GameLift's agent to enable further communication.
const initSDKOutcome = GameLiftServerAPI.InitSDK();
if (!initSDKOutcome.Success) {
    console.log("GameLiftServerAPI.InitSDK failure: " + initSDKOutcome.Error.toString());
    return;
}

const processParameters = new GameLift.ProcessParameters();
processParameters.OnStartGameSession = (session) => {
    // When a game session is created, GameLift sends an activation request to the game server and passes along the game session object containing game properties and other settings.
    // Here is where a game server should take action based on the game session object.
    // Once the game server is ready to receive incoming player connections, it should invoke GameLiftServerAPI.ActivateGameSession()
    console.log("GameLiftServerAPI initiating StartGameSession: " + Stringize(session));
    gameServerRunner.OnStartGameSession(session, () => {
        GameLiftServerAPI.ActivateGameSession();
        console.log("GameLiftServerAPI completed StartGameSession");
    });
};

processParameters.OnUpdateGameSession = (session) => {
    // When a game session is updated (e.g. by FlexMatch backfill), GameLiftsends a request to the game
    // server containing the updated game session object.  The game server can then examine the provided
    // matchmakerData and handle new incoming players appropriately.
    // updateReason is the reason this update is being supplied.
    console.log("GameLiftServerAPI received OnUpdateGameSession: " + Stringize(session));
    gameServerRunner.OnUpdateGameSession(session);
};

processParameters.OnProcessTerminate = () => {
    // OnProcessTerminate callback. GameLift will invoke this callback before shutting down an instance hosting this game server.
    // It gives this game server a chance to save its state, communicate with services, etc., before being shut down.
    // In this case, we simply tell GameLift we are indeed going to shutdown.
    console.log("GameLiftServerAPI initiating OnProcessTerminate");
    gameServerRunner.OnProcessTerminate(() => {
        console.log("GameLiftServerAPI finalizing OnProcessTerminate");
        GameLiftServerAPI.ProcessEnding();
        console.log("GameLiftServerAPI completed OnProcessTerminate");
    });
};

processParameters.OnHealthCheck = () => {
    // This is the HealthCheck callback.
    // GameLift will invoke this callback every 60 seconds or so.
    // Here, a game server might want to check the health of dependencies and such.
    // Simply return true if healthy, false otherwise.
    // The game server has 60 seconds to respond with its health status. GameLift will default to 'false' if the game server doesn't respond in time.
    // In this case, we're always healthy!
    // console.log("GameLiftServerAPI health check");
    return true;
};

// Here, the game server tells GameLift what set of files to upload when the game session ends.
// GameLift will upload everything specified here for the developers to fetch later.
processParameters.LogParameters = new GameLift.LogParameters([
    '/local/game/logs'
]);

// Initiate GameLift startup
(async () => {
    try {
        // Tell GameLift which port will listen for incoming player connections
        processParameters.Port = await gameServerRunner.run_async();
        console.log(`Port advertised to GameLift: ${processParameters.Port}`);

        // Calling ProcessReady tells GameLift this game server is ready to receive incoming game sessions
        const processReadyOutcome = await GameLiftServerAPI.ProcessReady(processParameters);
        if (processReadyOutcome.Success) {
            console.log("GameLiftServerAPI.ProcessReady success");
        } else {
            console.log("GameLiftServerAPI.ProcessReady failure: " + processReadyOutcome.Error.toString());
        }
    } catch (e) {
        console.log(e);
    }
})();
