/* eslint no-undef: 0 */
(function (global, factory) {
    typeof exports === 'object' && typeof module !== 'undefined' ? module.exports = factory() :
        typeof define === 'function' && define.amd ? define(factory) :
            (global.GameSparksRT = factory());
}(this, (function () { 'use strict';

    var Constants = {

        MAX_RTDATA_SLOTS: 128,
        TCP_CONNECT_TIMEOUT_SECONDS: 5,

        connectState: {
            DISCONNECTED: 0,
            CONNECTING: 1,
            RELIABLE_ONLY: 2,
            RELIABLE_AND_FAST_SEND: 3,
            RELIABLE_AND_FAST: 4
        },

        deliveryIntent: {
            RELIABLE: 0,
            UNRELIABLE: 1,
            UNRELIABLE_SEQUENCED: 2
        }

    };

    var _typeof = typeof Symbol === "function" && typeof Symbol.iterator === "symbol" ? function (obj) {
        return typeof obj;
    } : function (obj) {
        return obj && typeof Symbol === "function" && obj.constructor === Symbol && obj !== Symbol.prototype ? "symbol" : typeof obj;
    };

    var classCallCheck = function (instance, Constructor) {
        if (!(instance instanceof Constructor)) {
            throw new TypeError("Cannot call a class as a function");
        }
    };

    var createClass = function () {
        function defineProperties(target, props) {
            for (var i = 0; i < props.length; i++) {
                var descriptor = props[i];
                descriptor.enumerable = descriptor.enumerable || false;
                descriptor.configurable = true;
                if ("value" in descriptor) descriptor.writable = true;
                Object.defineProperty(target, descriptor.key, descriptor);
            }
        }

        return function (Constructor, protoProps, staticProps) {
            if (protoProps) defineProperties(Constructor.prototype, protoProps);
            if (staticProps) defineProperties(Constructor, staticProps);
            return Constructor;
        };
    }();

    var defineProperty = function (obj, key, value) {
        if (key in obj) {
            Object.defineProperty(obj, key, {
                value: value,
                enumerable: true,
                configurable: true,
                writable: true
            });
        } else {
            obj[key] = value;
        }

        return obj;
    };

    var get = function get(object, property, receiver) {
        if (object === null) object = Function.prototype;
        var desc = Object.getOwnPropertyDescriptor(object, property);

        if (desc === undefined) {
            var parent = Object.getPrototypeOf(object);

            if (parent === null) {
                return undefined;
            } else {
                return get(parent, property, receiver);
            }
        } else if ("value" in desc) {
            return desc.value;
        } else {
            var getter = desc.get;

            if (getter === undefined) {
                return undefined;
            }

            return getter.call(receiver);
        }
    };

    var inherits = function (subClass, superClass) {
        if (typeof superClass !== "function" && superClass !== null) {
            throw new TypeError("Super expression must either be null or a function, not " + typeof superClass);
        }

        subClass.prototype = Object.create(superClass && superClass.prototype, {
            constructor: {
                value: subClass,
                enumerable: false,
                writable: true,
                configurable: true
            }
        });
        if (superClass) Object.setPrototypeOf ? Object.setPrototypeOf(subClass, superClass) : subClass.__proto__ = superClass;
    };

    var possibleConstructorReturn = function (self, call) {
        if (!self) {
            throw new ReferenceError("this hasn't been initialised - super() hasn't been called");
        }

        return call && (typeof call === "object" || typeof call === "function") ? call : self;
    };

    var _logLevelStrings;

    // Need to define levels first so they can be used in the static properties.
    var level = {
        DEBUG: 0,
        INFO: 1,
        WARN: 2,
        ERROR: 3
    };

    var Logger = {

        level: level,

        logLevelStrings: (_logLevelStrings = {}, defineProperty(_logLevelStrings, level.DEBUG, 'DEBUG'), defineProperty(_logLevelStrings, level.INFO, 'INFO'), defineProperty(_logLevelStrings, level.WARN, 'WARN'), defineProperty(_logLevelStrings, level.ERROR, 'ERROR'), _logLevelStrings),

        currLogLevel: level.INFO,

        tagLevels: {},

        setRootLogLevel: function setRootLogLevel(level) {
            this.currLogLevel = level;
        },

        setLogLevel: function setLogLevel(tag, level) {
            this.tagLevels[tag] = level;
        },

        shouldLog: function shouldLog(tag, level) {
            for (var key in this.tagLevels) {
                var value = this.tagLevels[key];

                if (key == tag) {
                    return value >= level;
                }
            }

            return this.currLogLevel <= level;
        },

        log: function log(tag, level, msg) {
            if (this.shouldLog(tag, level)) {
                console.log('[' + this.logLevelStrings[level] + '] ' + tag + ': ' + msg); // eslint-disable-line no-console
            }
        },
        debug: function debug(tag, msg) {
            this.log(tag, this.level.DEBUG, msg);
        },
        info: function info(tag, msg) {
            this.log(tag, this.level.INFO, msg);
        },
        warn: function warn(tag, msg) {
            this.log(tag, this.level.WARN, msg);
        },
        error: function error(tag, msg) {
            this.log(tag, this.level.ERROR, msg);
        }
    };

    var ObjectPool = function () {
        function ObjectPool(creator, refresher, maxSize) {
            classCallCheck(this, ObjectPool);

            this.stack = [];
            this.creator = creator;
            this.refresher = refresher;
            this.maxSize = maxSize;
        }

        createClass(ObjectPool, [{
            key: "pop",
            value: function pop() {
                if (this.stack.length == 0) {
                    return this.creator();
                } else {
                    return this.stack.shift();
                }
            }
        }, {
            key: "push",
            value: function push(item) {
                if (item) {
                    if (this.stack.indexOf(item) >= 0) {
                        return;
                    }
                    if (this.stack.length < this.maxSize) {
                        if (this.refresher) {
                            this.refresher(item);
                        }
                        this.stack.push(item);
                    }
                }
            }
        }, {
            key: "dispose",
            value: function dispose() {
                this.stack = [];
            }
        }]);
        return ObjectPool;
    }();

    var ActionCommand = function () {
        function ActionCommand() {
            classCallCheck(this, ActionCommand);

            this.action = null;
        }

        createClass(ActionCommand, [{
            key: 'configure',
            value: function configure(action) {
                this.action = action;

                return this;
            }
        }, {
            key: 'execute',
            value: function execute() {
                this.action();
                ActionCommand.pool.push(this);
            }
        }]);
        return ActionCommand;
    }();

    ActionCommand.pool = new ObjectPool(function () {
        return new ActionCommand();
    }, null, 5);

    var LogCommand = function () {
        function LogCommand() {
            classCallCheck(this, LogCommand);

            this.tag = null;
            this.msg = null;
            this.level = null;
            this.session = null;
        }

        createClass(LogCommand, [{
            key: 'configure',
            value: function configure(session, tag, level, msg) {
                this.tag = tag;
                this.msg = msg;
                this.level = level;
                this.session = session;

                return this;
            }
        }, {
            key: 'execute',
            value: function execute() {
                if (Logger.shouldLog(this.tag, this.level)) {
                    if (this.session.peerId) {
                        Logger.log(this.tag, this.level, '[' + this.session.peerId + '] ' + this.msg);
                    } else {
                        Logger.log(this.tag, this.level, this.msg);
                    }
                }

                LogCommand.pool.push(this);
            }
        }]);
        return LogCommand;
    }();

    LogCommand.pool = new ObjectPool(function () {
        return new LogCommand();
    }, null, 5);

    var OpCodes = {
        LOGIN_RESULT: -1,
        PING_RESULT: -3,
        UDP_CONNECT_MESSAGE: -5,
        PLAYER_READY_MESSAGE: -7,
        PLAYER_CONNECT_MESSAGE: -101,
        PLAYER_DISCONNECT_MESSAGE: -103
    };

    var RTRequest = function () {
        function RTRequest() {
            classCallCheck(this, RTRequest);

            this.data = null;
            this.opCode = 0;
            this.targetPlayers = [];
            this.intent = Constants.deliveryIntent.RELIABLE;
        }

        createClass(RTRequest, [{
            key: 'toPacket',
            value: function toPacket() /*session, fast*/{
                return null;
            }
        }, {
            key: 'reset',
            value: function reset() {
                this.targetPlayers = [];
            }
        }, {
            key: 'serialize',
            value: function serialize() /*stream*/{}
        }]);
        return RTRequest;
    }();

    var CustomRequest = function (_RTRequest) {
        inherits(CustomRequest, _RTRequest);

        function CustomRequest() {
            classCallCheck(this, CustomRequest);

            var _this = possibleConstructorReturn(this, (CustomRequest.__proto__ || Object.getPrototypeOf(CustomRequest)).call(this));

            _this.payload = null;
            return _this;
        }

        createClass(CustomRequest, [{
            key: 'configure',
            value: function configure(opCode, intent, payload, data, targetPlayers) {
                this.opCode = opCode;
                this.payload = payload;
                this.intent = intent;
                this.data = data;

                if (targetPlayers != null) {
                    for (var i in targetPlayers) {
                        this.targetPlayers.push(targetPlayers[i]);
                    }
                }
            }
        }, {
            key: 'toPacket',
            value: function toPacket(session, fast) {
                var p = PooledObjects.packetPool.pop();

                p.opCode = this.opCode;
                p.data = this.data;
                p.session = session;

                if (!fast && this.intent != Constants.deliveryIntent.RELIABLE) {
                    p.reliable = false;
                }

                if (this.intent == Constants.deliveryIntent.UNRELIABLE_SEQUENCED) {
                    p.sequenceNumber = session.nextSequenceNumber();
                }

                if (this.targetPlayers.length > 0) {
                    p.targetPlayers = this.targetPlayers;
                }

                p.request = this;

                return p;
            }
        }, {
            key: 'serialize',
            value: function serialize(stream) {
                if (this.payload) {
                    stream.writeBytes(this.payload, 0, this.payload.length);
                }
            }
        }, {
            key: 'reset',
            value: function reset() {
                this.payload = null;
                get(CustomRequest.prototype.__proto__ || Object.getPrototypeOf(CustomRequest.prototype), 'reset', this).call(this);
            }
        }]);
        return CustomRequest;
    }(RTRequest);

    var PositionStream = function () {
        function PositionStream() {
            classCallCheck(this, PositionStream);

            this.tempBuffer = [];
            this.bytesRead = 0;
            this.stream = null;
        }

        createClass(PositionStream, [{
            key: "wrap",
            value: function wrap(baseStream) {
                this.bytesRead = 0;
                this.stream = baseStream;
            }
        }, {
            key: "read",
            value: function read(buffer, offset, count) {
                var ret = this.stream.read(buffer, offset, count);

                this.bytesRead = this.bytesRead + ret[0];

                return ret;
            }
        }, {
            key: "readByte",
            value: function readByte() {
                var ret = this.read(this.tempBuffer, 0, 1);

                if (ret[0] == 1) {
                    this.tempBuffer = ret[1];

                    return this.tempBuffer[0];
                }

                return -1;
            }
        }, {
            key: "seek",
            value: function seek(offset) {
                for (var i = 0; i < offset; i++) {
                    this.readByte();
                }

                return this.bytesRead;
            }
        }, {
            key: "getLength",
            value: function getLength() {
                return this.stream.getLength();
            }
        }, {
            key: "getPosition",
            value: function getPosition() {
                return this.bytesRead;
            }
        }]);
        return PositionStream;
    }();

    var LimitedPositionStream = function (_PositionStream) {
        inherits(LimitedPositionStream, _PositionStream);

        function LimitedPositionStream() {
            classCallCheck(this, LimitedPositionStream);

            var _this = possibleConstructorReturn(this, (LimitedPositionStream.__proto__ || Object.getPrototypeOf(LimitedPositionStream)).call(this));

            _this.limit = 0;
            return _this;
        }

        createClass(LimitedPositionStream, [{
            key: 'wrap',
            value: function wrap(baseStream, limit) {
                get(LimitedPositionStream.prototype.__proto__ || Object.getPrototypeOf(LimitedPositionStream.prototype), 'wrap', this).call(this, baseStream);

                this.limit = limit;
            }
        }, {
            key: 'read',
            value: function read(buffer, offset, count) {
                var toRead;

                if (count > this.limit - this.bytesRead) {
                    toRead = this.limit - this.bytesRead;
                } else {
                    toRead = count;
                }

                return get(LimitedPositionStream.prototype.__proto__ || Object.getPrototypeOf(LimitedPositionStream.prototype), 'read', this).call(this, buffer, offset, toRead);
            }
        }, {
            key: 'readByte',
            value: function readByte() {
                if (this.bytesRead >= this.limit) {
                    return -1;
                } else {
                    return get(LimitedPositionStream.prototype.__proto__ || Object.getPrototypeOf(LimitedPositionStream.prototype), 'readByte', this).call(this);
                }
            }
        }, {
            key: 'skipToEnd',
            value: function skipToEnd() {
                if (this.bytesRead < this.limit) {
                    var discardBytes = PooledObjects.byteBufferPool.pop();

                    while (this.read(discardBytes, this.bytesRead, 256)[0] == 256) {
                        // Deliberate no-op.
                    }

                    PooledObjects.byteBufferPool.push(discardBytes);
                }
            }
        }]);
        return LimitedPositionStream;
    }(PositionStream);

    var AbstractResult = function () {
        function AbstractResult() {
            classCallCheck(this, AbstractResult);

            this.abstractResultType = true;
            this.packet = null;
            this.session = null;
        }

        createClass(AbstractResult, [{
            key: "configure",
            value: function configure(packet, session) {
                this.packet = packet;
                this.session = session;
            }
        }, {
            key: "executeAsync",
            value: function executeAsync() {
                return true;
            }
        }]);
        return AbstractResult;
    }();

    var Key = function () {
        function Key(field, wireType) {
            classCallCheck(this, Key);

            this.field = field;
            this.wireType = wireType;
        }

        createClass(Key, [{
            key: 'toString',
            value: function toString() {
                return '[Key: ' + this.field + ', ' + this.wireType + ']';
            }
        }]);
        return Key;
    }();

    var Wire = {
        VARINT: 0,
        FIXED64: 1,
        LENGTH_DELIMITED: 2,
        FIXED32: 5
    };

    var ProtocolParser = {};

    ProtocolParser.readBool = function (stream) {
        var b = stream.readByte();

        if (b < 0) {
            Logger.warn('ProtocolParser', 'Stream ended too early');

            return false;
        } else if (b == 1) {
            return true;
        } else if (b == 0) {
            return false;
        }

        print('WARNING: Invalid boolean value');

        return false;
    };

    ProtocolParser.readUInt32 = function (stream) {
        var b;
        var val = 0;

        for (var n = 0; n <= 4; n++) {
            b = stream.readByte();
            if (b < 0) {
                Logger.warn('ProtocolParser', 'Stream ended too early');

                return 0;
            }

            if (n == 4 && (b & 0xF0) != 0) {
                Logger.warn('ProtocolParser', 'Got larger VarInt than 32bit unsigned');

                return 0;
            }

            if ((b & 0x80) == 0) {
                return val | b << 7 * n;
            }

            val = val | (b & 0x7F) << 7 * n;
        }

        Logger.warn('ProtocolParser', 'Got larger VarInt than 32bit unsigned');

        return 0;
    };

    ProtocolParser.readZInt32 = function (stream) {
        var val = ProtocolParser.readUInt32(stream);

        return val >> 1 ^ val << 31 >> 31;
    };

    ProtocolParser.readSingle = function (stream) {
        var bytes = [];
        var val = 0;

        for (var n = 1; n <= 4; n++) {
            bytes[4 - n] = stream.readByte();
        }

        val = bytes[0] << 24 | bytes[1] << 16 | (bytes[2] << 8 | bytes[3]);

        var negative = (bytes[0] & 0x80) == 0x80;
        var exponent = (val & 0x7f800000) >> 23;
        var sign;
        var mantissa = 0;

        if (negative) {
            sign = -1;
        } else {
            sign = 1;
        }

        if (exponent == 255) {
            if (negative) {
                value = Number.NEGATIVE_INFINITY;
            } else {
                value = Number.POSITIVE_INFINITY;
            }
        } else if (exponent == 0) {
            value = 0;
        } else {
            exponent = exponent - 127 + 1;

            for (var i = 3; i >= 2; i--) {
                mantissa += bytes[i];
                mantissa /= 256;
            }
            mantissa += bytes[1] & 0x7F;
            mantissa = (mantissa / 128 + 1) / 2.0;

            var value = sign * Math.pow(2.0, exponent) * mantissa;
        }

        return value;
    };

    ProtocolParser.readUInt64 = function (stream) {
        var b;
        var val = 0;

        for (var n = 0; n <= 9; n++) {
            b = stream.readByte();
            if (b < 0) {
                Logger.warn('ProtocolParser', 'Stream ended too early');

                return 0;
            }

            if (n == 9 && (b & 0xFE) != 0) {
                Logger.warn('ProtocolParser', 'Got larger VarInt than 64 bit unsigned');

                return 0;
            }

            if ((b & 0x80) == 0) {
                return val + b * Math.pow(128, n);
            }

            val = val + (b & 0x7F) * Math.pow(128, n);
        }

        Logger.warn('ProtocolParser', 'Got larger VarInt than 64 bit unsigned');

        return 0;
    };

    ProtocolParser.readZInt64 = function (stream) {
        var val = ProtocolParser.readUInt64(stream);

        if (val % 2 == 1) {
            return Math.floor(-val / 2);
        } else {
            return Math.floor(val / 2);
        }
    };

    ProtocolParser.readDouble = function (stream) {
        var bytes = [];
        var valh = 0;

        for (var n = 1; n <= 8; n++) {
            bytes[8 - n] = stream.readByte();
        }

        valh = bytes[0] << 24 | bytes[1] << 16;

        var negative = (bytes[0] & 0x80) == 0x80;
        var exponent = (valh & 0x7ff00000) >> 20;
        var mantissa = 0;
        var sign;

        if (negative) {
            sign = -1;
        } else {
            sign = 1;
        }

        if (exponent == 2047) {
            if (negative) {
                value = Number.NEGATIVE_INFINITY;
            } else {
                value = Number.POSITIVE_INFINITY;
            }
        } else if (exponent == 0) {
            value = 0;
        } else {
            exponent = exponent - 1023 + 1;

            for (var i = 7; i >= 2; i--) {
                mantissa += bytes[i];
                mantissa /= 256;
            }
            mantissa += bytes[1] & 0xF;
            mantissa = (mantissa / 16 + 1) / 2.0;

            var value = sign * Math.pow(2.0, exponent) * mantissa;
        }

        return value;
    };

    ProtocolParser.readString = function (stream) {
        var length = ProtocolParser.readUInt32(stream);
        var ms = PooledObjects.memoryStreamPool.pop();
        var buffer = PooledObjects.byteBufferPool.pop();
        var read = 0;
        var ret;
        var r;

        while (read < length) {
            ret = stream.read(buffer, 0, Math.min(length - read, buffer.length));
            r = ret[0];
            buffer = ret[1];

            if (r == 0) {
                Logger.warn('ProtocolParser', 'Expected ' + (length - read).toString() + ' got ' + read);

                return 0;
            }
            ms.writeBytes(buffer, 0, r);
            read = read + r;
        }

        ret = ms.toString().slice(0, ms.getPosition());

        PooledObjects.byteBufferPool.push(buffer);
        PooledObjects.memoryStreamPool.push(ms);

        return ret;
    };

    ProtocolParser.readKey = function (firstByte, stream) {
        if (firstByte < 128) {
            return new Key(firstByte >> 3, firstByte & 0x07);
        }

        var fieldID = ProtocolParser.readUInt32(stream) << 4 | firstByte >> 3 & 0x0F;

        return new Key(fieldID, firstByte & 0x07);
    };

    ProtocolParser.skipKey = function (stream, key) {
        if (key.wireType == Wire.FIXED32) {
            stream.seek(4);
        } else if (key.wireType == Wire.FIXED64) {
            stream.seek(8);
        } else if (key.wireType == Wire.LENGTH_DELIMITED) {
            stream.seek(ProtocolParser.readUInt32(stream));
        } else if (key.wireType == Wire.VARINT) {
            ProtocolParser.readSkipVarInt(stream);
        } else {
            Logger.warn('ProtocolParser', 'Unknown wire type: ' + key.wireType);
        }
    };

    ProtocolParser.readSkipVarInt = function (stream) {
        for (;;) {
            var b = stream.readByte();

            if (b < 0) {
                Logger.warn('ProtocolParser', 'Stream ended too early');

                return;
            }

            if ((b & 0x80) == 0) {
                return;
            }
        }
    };

    ProtocolParser.writeBool = function (stream, val) {
        if (val) {
            return stream.writeByte(1);
        } else {
            return stream.writeByte(0);
        }
    };

    ProtocolParser.writeUInt32 = function (stream, val) {
        var b;

        val = Math.abs(val);

        for (;;) {
            b = val & 0x7F;
            val = val >>> 7;
            if (val == 0) {
                stream.writeByte(b);

                break;
            } else {
                b = b | 0x80;

                stream.writeByte(b);
            }
        }
    };

    ProtocolParser.writeZInt32 = function (stream, val) {
        var val1 = val << 1;
        var val2 = val >> 31;

        ProtocolParser.writeUInt32(stream, val1 ^ val2);
    };

    ProtocolParser.writeUInt64 = function (stream, val) {
        var b;

        val = Math.abs(val);

        for (;;) {
            b = val & 0x7F;
            val = Math.floor(val / 128);
            if (val == 0) {
                stream.writeByte(b);

                break;
            } else {
                b = b | 0x80;

                stream.writeByte(b);
            }
        }
    };

    ProtocolParser.writeZInt64 = function (stream, val) {
        var sign = false;

        if (val < 0) {
            val = val + 1;

            val = -val;

            sign = true;
        }

        val = val * 2;

        if (sign == true) {
            val = val + 1;
        }

        ProtocolParser.writeUInt64(stream, val);
    };

    ProtocolParser.frexp = function (arg) {
        //  discuss at: http://locutus.io/c/frexp/
        // original by: Oskar Larsson Högfeldt (http://oskar-lh.name/)
        //      note 1: Instead of
        //      note 1: double frexp( double arg, int* exp );
        //      note 1: this is built as
        //      note 1: [double, int] frexp( double arg );
        //      note 1: due to the lack of pointers in JavaScript.
        //      note 1: See code comments for further information.
        //   example 1: frexp(1)
        //   returns 1: [0.5, 1]
        //   example 2: frexp(1.5)
        //   returns 2: [0.75, 1]
        //   example 3: frexp(3 * Math.pow(2, 500))
        //   returns 3: [0.75, 502]
        //   example 4: frexp(-4)
        //   returns 4: [-0.5, 3]
        //   example 5: frexp(Number.MAX_VALUE)
        //   returns 5: [0.9999999999999999, 1024]
        //   example 6: frexp(Number.MIN_VALUE)
        //   returns 6: [0.5, -1073]
        //   example 7: frexp(-Infinity)
        //   returns 7: [-Infinity, 0]
        //   example 8: frexp(-0)
        //   returns 8: [-0, 0]
        //   example 9: frexp(NaN)
        //   returns 9: [NaN, 0]
        // Potential issue with this implementation:
        // the precisions of Math.pow and the ** operator are undefined in the ECMAScript standard,
        // however, sane implementations should give the same results for Math.pow(2, <integer>) operations
        // Like frexp of C and std::frexp of C++,
        // but returns an array instead of using a pointer argument for passing the exponent result.
        // Object.is(n, frexp(n)[0] * 2 ** frexp(n)[1]) for all number values of n except when Math.isFinite(n) && Math.abs(n) > 2**1023
        // Object.is(n, (2 * frexp(n)[0]) * 2 ** (frexp(n)[1] - 1)) for all number values of n
        // Object.is(n, frexp(n)[0]) for these values of n: 0, -0, NaN, Infinity, -Infinity
        // Math.abs(frexp(n)[0]) is >= 0.5 and < 1.0 for any other number-type value of n
        // See http://en.cppreference.com/w/c/numeric/math/frexp for a more detailed description
        arg = Number(arg);
        var result = [arg, 0];
        if (arg !== 0 && Number.isFinite(arg)) {
            var absArg = Math.abs(arg);
            // Math.log2 was introduced in ES2015, use it when available
            var log2 = Math.log2 || function log2(n) {
                return Math.log(n) * Math.LOG2E;
            };
            var exp = Math.max(-1023, Math.floor(log2(absArg)) + 1);
            var x = absArg * Math.pow(2, -exp);
            // These while loops compensate for rounding errors that sometimes occur because of ECMAScript's Math.log2's undefined precision
            // and also works around the issue of Math.pow(2, -exp) === Infinity when exp <= -1024
            while (x < 0.5) {
                x *= 2;
                exp--;
            }
            while (x >= 1) {
                x *= 0.5;
                exp++;
            }
            if (arg < 0) {
                x = -x;
            }
            result[0] = x;
            result[1] = exp;
        }
        return result;
    };

    ProtocolParser.writeSingle = function (stream, val) {
        var bytes = [0, 0, 0, 0];
        var i;

        if (val != 0) {
            var anum = Math.abs(val);
            var ret = ProtocolParser.frexp(anum);
            var mantissa = ret[0];
            var exponent = ret[1];

            var sign = val != anum && 128 || 0;

            mantissa = mantissa * 2 - 1;
            if (mantissa == Infinity) {
                mantissa = 0;
                exponent = 255;
            } else {
                exponent = exponent - 1;
                exponent = exponent + 127;
                if (exponent < 0 || exponent >= 255) {
                    mantissa = 0;
                    if (exponent < 0) {
                        exponent = 0;
                    } else {
                        exponent = 255;
                    }
                }
            }

            bytes[0] = sign + (exponent >> 1);
            mantissa *= 128;

            var currentmantissa = Math.floor(mantissa);

            mantissa -= currentmantissa;
            bytes[1] = ((exponent & 0x1) << 7) + currentmantissa;
            for (i = 2; i < 4; i++) {
                mantissa *= 256;
                currentmantissa = Math.floor(mantissa);
                mantissa -= currentmantissa;
                bytes[i] = currentmantissa;
            }
        }

        for (i = bytes.length - 1; i >= 0; i--) {
            stream.writeByte(bytes[i]);
        }
    };

    ProtocolParser.writeDouble = function (stream, val) {
        var bytes = [0, 0, 0, 0, 0, 0, 0, 0];
        var i;

        if (val != 0) {
            var anum = Math.abs(val);
            var ret = ProtocolParser.frexp(anum);
            var mantissa = ret[0];
            var exponent = ret[1];

            var sign = val != anum && 128 || 0;

            mantissa = mantissa * 2 - 1;
            if (mantissa == Infinity) {
                mantissa = 0;
                exponent = 2047;
            } else {
                exponent = exponent - 1;
                exponent = exponent + 1023;
                if (exponent < 0 || exponent >= 2047) {
                    mantissa = 0;
                    if (exponent < 0) {
                        exponent = 0;
                    } else {
                        exponent = 2047;
                    }
                }
            }

            bytes[0] = sign + (exponent >> 4);
            mantissa *= 16;

            var currentmantissa = Math.floor(mantissa);

            mantissa -= currentmantissa;
            bytes[1] = ((exponent & 0xF) << 4) + currentmantissa;
            for (i = 2; i < 8; i++) {
                mantissa *= 256;
                currentmantissa = Math.floor(mantissa);
                mantissa -= currentmantissa;
                bytes[i] = currentmantissa;
            }
        }

        for (i = bytes.length - 1; i >= 0; i--) {
            stream.writeByte(bytes[i]);
        }
    };

    ProtocolParser.writeBytes = function (stream, val, len) {
        ProtocolParser.writeUInt32(stream, len);
        stream.writeBytes(val, 0, len);
    };

    ProtocolParser.writeString = function (stream, val) {
        var array = [];

        for (var i = 0; i < val.length; i++) {
            array.push(val.charCodeAt(i));
        }

        ProtocolParser.writeBytes(stream, array, val.length);
    };

    var LoginResult = function (_AbstractResult) {
        inherits(LoginResult, _AbstractResult);

        function LoginResult() {
            classCallCheck(this, LoginResult);

            var _this = possibleConstructorReturn(this, (LoginResult.__proto__ || Object.getPrototypeOf(LoginResult)).call(this));

            _this.success = false;
            _this.reconnectToken = null;
            _this.peerId = null;
            _this.activePeers = [];
            _this.fastPort = null;
            return _this;
        }

        createClass(LoginResult, [{
            key: 'execute',
            value: function execute() {
                this.session.connectToken = this.reconnectToken;
                this.session.peerId = this.peerId;

                if (this.packet.reliable == null || this.packet.reliable == true) {
                    if (this.fastPort != null && this.fastPort) {
                        this.session.fastPort = this.fastPort;
                    }

                    this.session.activePeers = this.activePeers.slice();

                    this.session.setConnectState(Constants.connectState.RELIABLE_ONLY);
                    this.session.log('LoginResult', Logger.level.DEBUG, 'TCP LoginResult, ActivePeers ' + this.session.activePeers.length);
                } else {
                    this.session.setConnectState(Constants.connectState.RELIABLE_AND_FAST_SEND);
                }

                LoginResult.pool.push(this);
            }
        }, {
            key: 'executeAsync',
            value: function executeAsync() {
                return false;
            }
        }]);
        return LoginResult;
    }(AbstractResult);

    LoginResult.pool = new ObjectPool(function () {
        return new LoginResult();
    }, function (instance) {
        instance.activePeers = [];
        instance.fastPort = null;
        instance.reconnectToken = null;
        instance.peerId = null;
    }, 5);

    LoginResult.deserialize = function (stream, instance) {
        if (instance.activePeers == null) {
            instance.activePeers = [];
        }

        for (;;) {
            var keyByte = stream.readByte();

            if (keyByte == -1) {
                break;
            }

            var _continue = false;

            if (keyByte == 8) {
                instance.success = ProtocolParser.readBool(stream);

                _continue = true;
            } else if (keyByte == 18) {
                instance.reconnectToken = ProtocolParser.readString(stream);

                _continue = true;
            } else if (keyByte == 24) {
                instance.peerId = ProtocolParser.readUInt64(stream);

                _continue = true;
            } else if (keyByte == 32) {
                instance.activePeers.push(ProtocolParser.readUInt64(stream));

                _continue = true;
            } else if (keyByte == 40) {
                instance.fastPort = ProtocolParser.readUInt64(stream);

                _continue = true;
            }

            if (!_continue) {
                var key = ProtocolParser.readKey(keyByte, stream);

                if (key.field == 0) {
                    Logger.warn('LoginResult', 'Invalid field id: 0, something went wrong in the stream');

                    return null;
                } else {
                    ProtocolParser.skipKey(stream, key);
                }
            }
        }

        return instance;
    };

    var PingResult = function (_AbstractResult) {
        inherits(PingResult, _AbstractResult);

        function PingResult() {
            classCallCheck(this, PingResult);
            return possibleConstructorReturn(this, (PingResult.__proto__ || Object.getPrototypeOf(PingResult)).apply(this, arguments));
        }

        createClass(PingResult, [{
            key: 'execute',
            value: function execute() {
                this.session.log('PingResult', Logger.level.DEBUG, '');

                PingResult.pool.push(this);
            }
        }, {
            key: 'executeAsync',
            value: function executeAsync() {
                return false;
            }
        }]);
        return PingResult;
    }(AbstractResult);

    PingResult.pool = new ObjectPool(function () {
        return new PingResult();
    }, null, 5);

    PingResult.deserialize = function (stream, instance) {
        for (;;) {
            var keyByte = stream.readByte();

            if (keyByte == -1) {
                break;
            }

            var key = ProtocolParser.readKey(keyByte, stream);

            if (key.field == 0) {
                Logger.warn('PingResult', 'Invalid field id: 0, something went wrong in the stream');

                return null;
            } else {
                ProtocolParser.skipKey(stream, key);
            }
        }

        return instance;
    };

    var UDPConnectMessage = function (_AbstractResult) {
        inherits(UDPConnectMessage, _AbstractResult);

        function UDPConnectMessage() {
            classCallCheck(this, UDPConnectMessage);
            return possibleConstructorReturn(this, (UDPConnectMessage.__proto__ || Object.getPrototypeOf(UDPConnectMessage)).apply(this, arguments));
        }

        createClass(UDPConnectMessage, [{
            key: 'execute',
            value: function execute() {
                var reliable;

                if (this.packet.reliable != null) {
                    reliable = this.packet.reliable;
                } else {
                    reliable = false;
                }

                this.session.log('UDPConnectMessage', Logger.level.DEBUG, '(UDP) reliable=' + reliable.toString() + ', ActivePeers ' + this.session.activePeers.length);

                if (!reliable) {
                    self.session.setConnectState(Constants.connectState.RELIABLE_AND_FAST);
                    self.session.sendData(-5, Constants.deliveryIntent.RELIABLE, null, null);
                } else {
                    self.session.log('UDPConnectMessage', Logger.level.DEBUG, 'TCP (Unexpected) UDPConnectMessage');
                }

                UDPConnectMessage.pool.push(this);
            }
        }, {
            key: 'executeAsync',
            value: function executeAsync() {
                return false;
            }
        }]);
        return UDPConnectMessage;
    }(AbstractResult);

    UDPConnectMessage.pool = new ObjectPool(function () {
        return new UDPConnectMessage();
    }, null, 5);

    UDPConnectMessage.deserialize = function (stream, instance) {
        for (;;) {
            var keyByte = stream.readByte();

            if (keyByte == -1) {
                break;
            }

            var key = ProtocolParser.readKey(keyByte, stream);

            if (key.field == 0) {
                Logger.warn('UDPConnectMessage', 'Invalid field id: 0, something went wrong in the stream');

                return null;
            } else {
                ProtocolParser.skipKey(stream, key);
            }
        }

        return instance;
    };

    var PlayerConnectMessage = function (_AbstractResult) {
        inherits(PlayerConnectMessage, _AbstractResult);

        function PlayerConnectMessage() {
            classCallCheck(this, PlayerConnectMessage);

            var _this = possibleConstructorReturn(this, (PlayerConnectMessage.__proto__ || Object.getPrototypeOf(PlayerConnectMessage)).call(this));

            _this.peerId = 0;
            _this.activePeers = [];
            return _this;
        }

        createClass(PlayerConnectMessage, [{
            key: 'execute',
            value: function execute() {
                this.session.activePeers = this.activePeers.slice();
                this.session.log('PlayerConnectMessage', Logger.level.DEBUG, 'PeerId=' + this.peerId + ', ActivePeers ' + this.session.activePeers.length);
                this.session.onPlayerConnect(this.peerId);

                PlayerConnectMessage.pool.push(this);
            }
        }]);
        return PlayerConnectMessage;
    }(AbstractResult);

    PlayerConnectMessage.pool = new ObjectPool(function () {
        return new PlayerConnectMessage();
    }, function (instance) {
        instance.activePeers = [];
    }, 5);

    PlayerConnectMessage.deserialize = function (stream, instance) {
        if (instance.activePeers == null) {
            instance.activePeers = [];
        }

        for (;;) {
            var keyByte = stream.readByte();

            if (keyByte == -1) {
                break;
            }

            var _continue = false;

            if (keyByte == 8) {
                instance.peerId = ProtocolParser.readUInt64(stream);

                _continue = true;
            } else if (keyByte == 32) {
                instance.activePeers.push(ProtocolParser.readUInt64(stream));

                _continue = true;
            }

            if (!_continue) {
                var key = ProtocolParser.readKey(keyByte, stream);

                if (key.field == 0) {
                    Logger.warn('PlayerConnectMessage', 'Invalid field id: 0, something went wrong in the stream');

                    return null;
                } else {
                    ProtocolParser.skipKey(stream, key);
                }
            }
        }

        return instance;
    };

    var PlayerDisconnectMessage = function (_AbstractResult) {
        inherits(PlayerDisconnectMessage, _AbstractResult);

        function PlayerDisconnectMessage() {
            classCallCheck(this, PlayerDisconnectMessage);

            var _this = possibleConstructorReturn(this, (PlayerDisconnectMessage.__proto__ || Object.getPrototypeOf(PlayerDisconnectMessage)).call(this));

            _this.peerId = 0;
            _this.activePeers = [];
            return _this;
        }

        createClass(PlayerDisconnectMessage, [{
            key: 'execute',
            value: function execute() {
                this.session.activePeers = [];
                this.session.activePeers = this.activePeers.slice();
                this.session.log('PlayerDisconnectMessage', Logger.level.DEBUG, 'PeerId=' + this.peerId + ', ActivePeers ' + this.session.activePeers.length);
                this.session.onPlayerDisconnect(this.peerId);

                PlayerDisconnectMessage.pool.push(this);
            }
        }]);
        return PlayerDisconnectMessage;
    }(AbstractResult);

    PlayerDisconnectMessage.pool = new ObjectPool(function () {
        return new PlayerDisconnectMessage();
    }, function (instance) {
        instance.activePeers = [];
    }, 5);

    PlayerDisconnectMessage.deserialize = function (stream, instance) {
        if (instance.activePeers == null) {
            instance.activePeers = [];
        }

        for (;;) {
            var keyByte = stream.readByte();

            if (keyByte == -1) {
                break;
            }

            var _continue = false;

            if (keyByte == 8) {
                instance.peerId = ProtocolParser.readUInt64(stream);

                _continue = true;
            } else if (keyByte == 32) {
                instance.activePeers.push(ProtocolParser.readUInt64(stream));

                _continue = true;
            }

            if (!_continue) {
                var key = ProtocolParser.readKey(keyByte, stream);

                if (key.field == 0) {
                    Logger.warn('PlayerDisconnectMessage', 'Invalid field id: 0, something went wrong in the stream');

                    return null;
                } else {
                    ProtocolParser.skipKey(stream, key);
                }
            }
        }

        return instance;
    };

    var CommandFactory = {
        getCommand: function getCommand(opCode, sender, sequence, stream, session, data, packetSize) {
            var ret = null;
            var limit = ProtocolParser.readUInt32(stream);
            var lps = PooledObjects.limitedPositionStreamPool.pop();

            lps.wrap(stream, limit);

            if (opCode == OpCodes.LOGIN_RESULT) {
                ret = LoginResult.deserialize(lps, LoginResult.pool.pop());
            } else if (opCode == OpCodes.PING_RESULT) {
                ret = PingResult.deserialize(lps, PingResult.pool.pop());
            } else if (opCode == OpCodes.UDP_CONNECT_MESSAGE) {
                ret = UDPConnectMessage.deserialize(lps, UDPConnectMessage.pool.pop());
            } else if (opCode == OpCodes.PLAYER_CONNECT_MESSAGE) {
                ret = PlayerConnectMessage.deserialize(lps, PlayerConnectMessage.pool.pop());
            } else if (opCode == OpCodes.PLAYER_DISCONNECT_MESSAGE) {
                ret = PlayerDisconnectMessage.deserialize(lps, PlayerDisconnectMessage.pool.pop());
            } else {
                if (session.shouldExecute(sender, sequence)) {
                    ret = CustomCommand.pool.pop().configure(opCode, sender, lps, data, limit, session, packetSize);
                }
            }

            lps.skipToEnd();

            PooledObjects.limitedPositionStreamPool.push(lps);

            return ret;
        }
    };

    var RTVector = function RTVector(x, y, z, w) {
        classCallCheck(this, RTVector);

        this.x = x;
        this.y = y;
        this.z = z;
        this.w = w;
    };

    var RTDataSerializer = {};

    RTDataSerializer.cache = new ObjectPool(function () {
        return new RTData();
    }, function (rtData) {
        for (var i = 0; i < rtData.data.length; i++) {
            if (rtData.data[i].data_val) {
                rtData.data[i].data_val.dispose();
            }
            rtData.data[i].reset();
        }
    }, 5);

    RTDataSerializer.get = function () {
        return RTDataSerializer.cache.pop();
    };

    RTDataSerializer.readRTData = function (stream, instance) {
        if (instance == null) {
            instance = RTDataSerializer.cache.pop();
        }

        var limit = ProtocolParser.readUInt32(stream);

        limit = limit + stream.getPosition();

        for (;;) {
            if (stream.getPosition() >= limit) {
                if (stream.getPosition() == limit) {
                    break;
                } else {
                    Logger.warn('RTDataSerializer', 'Read past max limit');

                    return null;
                }
            }

            var keyByte = stream.readByte();

            if (keyByte == -1) {
                break;
            }

            var key = ProtocolParser.readKey(keyByte, stream);

            if (key.wireType == Wire.VARINT) {
                instance.data[key.field] = RTVal.newLong(ProtocolParser.readZInt64(stream));
            } else if (key.wireType == Wire.FIXED32) {
                instance.data[key.field] = RTVal.newFloat(ProtocolParser.readSingle(stream));
            } else if (key.wireType == Wire.FIXED64) {
                instance.data[key.field] = RTVal.newDouble(ProtocolParser.readDouble(stream));
            } else if (key.wireType == Wire.LENGTH_DELIMITED) {
                instance.data[key.field] = RTVal.deserializeLengthDelimited(stream);
            }

            if (key.field == 0) {
                Logger.warn('RTDataSerializer', 'Invalid field id: 0, something went wrong in the stream');

                return null;
            }
        }

        return instance;
    };

    RTDataSerializer.writeRTData = function (stream, instance) {
        var ms = PooledObjects.memoryStreamPool.pop();

        for (var index = 1; index < instance.data.length; index++) {
            var entry = instance.data[index];
            if (entry.long_val != null) {
                ProtocolParser.writeUInt32(ms, index << 3);

                ProtocolParser.writeZInt64(ms, entry.long_val);
            } else if (entry.float_val != null) {
                ProtocolParser.writeUInt32(ms, index << 3 | 5);

                ProtocolParser.writeSingle(ms, entry.float_val);
            } else if (entry.double_val != null) {
                ProtocolParser.writeUInt32(ms, index << 3 | 1);

                ProtocolParser.writeDouble(ms, entry.double_val);
            } else if (entry.data_val || entry.string_val || entry.vec_val) {
                ProtocolParser.writeUInt32(ms, index << 3 | 2);

                entry.serializeLengthDelimited(ms);
            }
        }

        var buffer = ms.getBuffer();

        ProtocolParser.writeBytes(stream, buffer, ms.getPosition());

        PooledObjects.memoryStreamPool.push(ms);
    };

    var RTVal = function () {
        function RTVal() {
            classCallCheck(this, RTVal);

            this.long_val = null;
            this.float_val = null;
            this.double_val = null;
            this.data_val = null;
            this.string_val = null;
            this.vec_val = null;
        }

        createClass(RTVal, [{
            key: 'serializeLengthDelimited',
            value: function serializeLengthDelimited(stream) {
                var ms = PooledObjects.memoryStreamPool.pop();

                if (this.string_val) {
                    ms.writeByte(10);
                    ProtocolParser.writeString(ms, this.string_val);
                } else if (this.data_val) {
                    ms.writeByte(114);
                    RTData.writeRTData(ms, this.data_val);
                } else if (this.vec_val) {
                    var vec_value = this.vec_val;
                    var numberOfFloatsSet = 0;

                    ms.writeByte(18);

                    if (vec_value.x != null) {
                        numberOfFloatsSet = numberOfFloatsSet + 1;
                    }
                    if (vec_value.y != null) {
                        numberOfFloatsSet = numberOfFloatsSet + 1;
                    }
                    if (vec_value.z != null) {
                        numberOfFloatsSet = numberOfFloatsSet + 1;
                    }
                    if (vec_value.w != null) {
                        numberOfFloatsSet = numberOfFloatsSet + 1;
                    }

                    ProtocolParser.writeUInt32(ms, 4 * numberOfFloatsSet);

                    for (var i = 1; i <= numberOfFloatsSet; i++) {
                        if (i == 1) {
                            ProtocolParser.writeSingle(ms, vec_value.x);
                        } else if (i == 2) {
                            ProtocolParser.writeSingle(ms, vec_value.y);
                        } else if (i == 3) {
                            ProtocolParser.writeSingle(ms, vec_value.z);
                        } else if (i == 4) {
                            ProtocolParser.writeSingle(ms, vec_value.w);
                        }
                    }
                }

                var data = ms.getBuffer();

                ProtocolParser.writeBytes(stream, data, ms.getPosition());

                PooledObjects.memoryStreamPool.push(ms);
            }
        }, {
            key: 'reset',
            value: function reset() {
                if (this.data_val) {
                    this.data_val.dispose();
                }
                this.long_val = null;
                this.float_val = null;
                this.double_val = null;
                this.data_val = null;
                this.string_val = null;
                this.vec_val = null;
            }
        }, {
            key: 'dirty',
            value: function dirty() {
                if (this.long_val != null) {
                    return true;
                } else if (this.float_val != null) {
                    return true;
                } else if (this.double_val != null) {
                    return true;
                } else if (this.data_val) {
                    return true;
                } else if (this.string_val) {
                    return true;
                } else if (this.vec_val) {
                    return true;
                }

                return false;
            }
        }, {
            key: 'asString',
            value: function asString() {
                if (this.long_val != null) {
                    return this.long_val.toString();
                } else if (this.float_val != null) {
                    return this.float_val.toString();
                } else if (this.double_val != null) {
                    return this.double_val.toString();
                } else if (this.data_val) {
                    return this.data_val.toString();
                } else if (this.string_val) {
                    return '"' + this.string_val + '"';
                } else if (this.vec_val) {
                    var ret = '|';

                    if (this.vec_val.x != null) {
                        ret = ret + this.vec_val.x.toString() + '|';
                    }
                    if (this.vec_val.y != null) {
                        ret = ret + this.vec_val.y.toString() + '|';
                    }
                    if (this.vec_val.z != null) {
                        ret = ret + this.vec_val.z.toString() + '|';
                    }
                    if (this.vec_val.w != null) {
                        ret = ret + this.vec_val.w.toString() + '|';
                    }

                    return ret;
                }

                return null;
            }
        }]);
        return RTVal;
    }();

    RTVal.newLong = function (value) {
        var instance = new RTVal();

        instance.long_val = value;

        return instance;
    };

    RTVal.newFloat = function (value) {
        var instance = new RTVal();

        instance.float_val = value;

        return instance;
    };

    RTVal.newDouble = function (value) {
        var instance = new RTVal();

        instance.double_val = value;

        return instance;
    };

    RTVal.newRTData = function (value) {
        var instance = new RTVal();

        instance.data_val = value;

        return instance;
    };

    RTVal.newString = function (value) {
        var instance = new RTVal();

        instance.string_val = value;

        return instance;
    };

    RTVal.newRTVector = function (value) {
        var instance = new RTVal();

        instance.vec_val = value;

        return instance;
    };

    RTVal.deserializeLengthDelimited = function (stream) {
        var instance = new RTVal();

        var limit = ProtocolParser.readUInt32(stream);

        limit = limit + stream.getPosition();

        for (;;) {
            if (stream.getPosition() >= limit) {
                if (stream.getPosition() == limit) {
                    break;
                } else {
                    Logger.warn('RTVal', 'Read past max limit');

                    return 0;
                }
            }

            var keyByte = stream.readByte();

            if (keyByte == -1) {
                Logger.warn('RTVal', 'End of stream');

                return 0;
            }

            var _continue = false;

            if (keyByte == 10) {
                instance.string_val = ProtocolParser.readString(stream);

                _continue = true;
            } else if (keyByte == 18) {
                var end2 = ProtocolParser.readUInt32(stream);

                end2 = end2 + stream.getPosition();

                var v = new RTVector();

                var i = 0;
                while (stream.getPosition() < end2) {
                    var read = ProtocolParser.readSingle(stream);

                    if (i == 0) {
                        v.x = read;
                    } else if (i == 1) {
                        v.y = read;
                    } else if (i == 2) {
                        v.z = read;
                    } else if (i == 3) {
                        v.w = read;
                    }

                    i = i + 1;
                }
                instance.vec_val = v;

                if (stream.getPosition() != end2) {
                    Logger.warn('RTVal', 'Read too many bytes in packed data');

                    return null;
                }

                _continue = true;
            } else if (keyByte == 114) {
                if (instance.data_val == null) {
                    instance.data_val = RTDataSerializer.cache.pop();
                }
                RTData.readRTData(stream, instance.data_val);

                _continue = true;
            }

            if (!_continue) {
                var key = ProtocolParser.readKey(keyByte, stream);

                if (key.field == 0) {
                    Logger.warn('RTVal', 'Invalid field id: 0, something went wrong in the stream');

                    return null;
                } else {
                    ProtocolParser.skipKey(stream, key);
                }
            }
        }

        return instance;
    };

    var RTData = function () {
        function RTData() {
            classCallCheck(this, RTData);

            this.data = [];
            for (var i = 0; i < Constants.MAX_RTDATA_SLOTS; i++) {
                this.data.push(new RTVal());
            }
        }

        createClass(RTData, [{
            key: 'dispose',
            value: function dispose() {
                for (var i = 0; i < this.data.length; i++) {
                    if (this.data[i].dirty()) {
                        this.data[i] = new RTVal();
                    }
                }

                RTDataSerializer.cache.push(this);
            }
        }, {
            key: 'getRTVector',
            value: function getRTVector(index) {
                return this.data[index].vec_val;
            }
        }, {
            key: 'getLong',
            value: function getLong(index) {
                return this.data[index].long_val;
            }
        }, {
            key: 'getFloat',
            value: function getFloat(index) {
                return this.data[index].float_val;
            }
        }, {
            key: 'getDouble',
            value: function getDouble(index) {
                return this.data[index].double_val;
            }
        }, {
            key: 'getString',
            value: function getString(index) {
                return this.data[index].string_val;
            }
        }, {
            key: 'getData',
            value: function getData(index) {
                return this.data[index].data_val;
            }
        }, {
            key: 'setRTVector',
            value: function setRTVector(index, value) {
                this.data[index] = RTVal.newRTVector(value);

                return this;
            }
        }, {
            key: 'setLong',
            value: function setLong(index, value) {
                if (isFinite(value)) {
                    this.data[index] = RTVal.newLong(value);
                } else {
                    Logger.error('RTData', 'Not a valid number error');
                }

                return this;
            }
        }, {
            key: 'setFloat',
            value: function setFloat(index, value) {
                if (isFinite(value)) {
                    this.data[index] = RTVal.newFloat(value);
                } else {
                    Logger.error('RTData', 'Not a valid number error');
                }

                return this;
            }
        }, {
            key: 'setDouble',
            value: function setDouble(index, value) {
                if (isFinite(value)) {
                    this.data[index] = RTVal.newDouble(value);
                } else {
                    Logger.error('RTData', 'Not a valid number error');
                }

                return this;
            }
        }, {
            key: 'setString',
            value: function setString(index, value) {
                this.data[index] = RTVal.newString(value);

                return this;
            }
        }, {
            key: 'setData',
            value: function setData(index, value) {
                this.data[index] = RTVal.newRTData(value);

                return this;
            }
        }, {
            key: 'toString',
            value: function toString() {
                return this.asString();
            }
        }, {
            key: 'asString',
            value: function asString() {
                var builder = ' {';

                for (var i = 0; i < Constants.MAX_RTDATA_SLOTS; i++) {
                    var val = this.data[i].asString();

                    if (val != null) {
                        builder = builder + ' [' + i.toString() + ' ' + val + '] ';
                    }
                }
                builder = builder + '} ';

                return builder;
            }
        }]);
        return RTData;
    }();

    RTData.get = function () {
        return RTDataSerializer.cache.pop();
    };

    RTData.readRTData = function (stream, instance) {
        return RTDataSerializer.readRTData(stream, instance);
    };

    RTData.writeRTData = function (stream, instance) {
        RTDataSerializer.writeRTData(stream, instance);
    };

    var Packet = function () {
        function Packet() {
            classCallCheck(this, Packet);

            this.opCode = 0;
            this.sequenceNumber = null;
            this.requestId = null;
            this.targetPlayers = null;
            this.sender = null;
            this.reliable = null;
            this.data = null;
            this.payload = null;
            this.request = null;
            this.hasPayload = false;
            this.command = null;
            this.session = null;
        }

        createClass(Packet, [{
            key: 'reset',
            value: function reset() {
                this.opCode = 0;
                this.sequenceNumber = null;
                this.requestId = null;
                this.targetPlayers = null;
                this.sender = null;
                this.reliable = null;
                this.payload = null;
                this.command = null;
                this.request = null;
                this.hasPayload = false;
                this.data = null;
            }
        }, {
            key: 'toString',
            value: function toString() {
                return '{OpCode:' + this.opCode + ',TargetPlayers:' + this.targetToString() + '}';
            }
        }, {
            key: 'targetToString',
            value: function targetToString() {
                var s = '[';

                if (this.targetPlayers != null) {
                    for (var i = 0; i < this.targetPlayers.length; i++) {
                        s = s + this.targetPlayers[i] + ' ';
                    }
                }

                return s + ']';
            }
        }, {
            key: 'readPayload',
            value: function readPayload(stream, packetSize) {
                this.hasPayload = true;
                if (this.sender != null) {
                    this.command = CommandFactory.getCommand(this.opCode, this.sender, this.sequenceNumber, stream, this.session, this.data, packetSize);
                } else {
                    this.command = CommandFactory.getCommand(this.opCode, 0, this.sequenceNumber, stream, this.session, this.data, packetSize);
                }

                return null;
            }
        }, {
            key: 'writePayload',
            value: function writePayload(stream) {
                if (this.request != null) {
                    var ms = PooledObjects.memoryStreamPool.pop();

                    this.request.serialize(ms);

                    var written = ms.getBuffer();

                    if (ms.getPosition() > 0) {
                        stream.writeByte(122);
                        ProtocolParser.writeBytes(stream, written, ms.getPosition());
                    }

                    PooledObjects.memoryStreamPool.push(ms);
                } else if (this.payload != null) {
                    stream.writeByte(122);
                    ProtocolParser.writeBytes(stream, this.payload, this.payload.length);
                }
            }
        }]);
        return Packet;
    }();

    Packet.serialize = function (stream, instance) {
        stream.writeByte(8);

        ProtocolParser.writeZInt32(stream, instance.opCode);

        if (instance.sequenceNumber != null) {
            stream.writeByte(16);
            ProtocolParser.writeUInt64(stream, instance.sequenceNumber);
        }

        if (instance.requestId != null) {
            stream.writeByte(24);
            ProtocolParser.writeUInt64(stream, instance.requestId);
        }

        if (instance.targetPlayers != null) {
            for (var i = 0; i < instance.targetPlayers.length; i++) {
                stream.writeByte(32);
                ProtocolParser.writeUInt64(stream, instance.targetPlayers[i]);
            }
        }

        if (instance.sender != null) {
            stream.writeByte(40);
            ProtocolParser.writeUInt64(stream, instance.sender);
        }

        if (instance.reliable != null) {
            stream.writeByte(48);
            ProtocolParser.writeBool(stream, instance.reliable);
        }

        if (instance.data != null) {
            stream.writeByte(114);
            RTData.writeRTData(stream, instance.data);
        }

        instance.writePayload(stream);

        return stream;
    };

    Packet.serializeLengthDelimited = function (stream, instance) {
        var ms = PooledObjects.memoryStreamPool.pop();
        var ret;

        Packet.serialize(ms, instance);

        var data = ms.getBuffer();

        ProtocolParser.writeBytes(stream, data, ms.getPosition());

        ret = ms.getPosition();

        PooledObjects.memoryStreamPool.push(ms);

        return ret;
    };

    Packet.deserializeLengthDelimited = function (stream, instance) {
        var limit = ProtocolParser.readUInt32(stream);
        var origLimit = limit;

        limit = limit + stream.getPosition();

        for (;;) {
            if (stream.getPosition() >= limit) {
                if (stream.getPosition() == limit) {
                    break;
                } else {
                    Logger.warn('Packet', 'Read past max limit');

                    return 0;
                }
            }

            var keyByte = stream.readByte();

            if (keyByte == -1) {
                Logger.warn('Packet', 'End of stream');

                return 0;
            }

            var _continue = false;

            if (keyByte == 8) {
                instance.opCode = ProtocolParser.readZInt32(stream);

                _continue = true;
            } else if (keyByte == 16) {
                instance.sequenceNumber = ProtocolParser.readUInt64(stream);

                _continue = true;
            } else if (keyByte == 24) {
                instance.requestId = ProtocolParser.readUInt64(stream);

                _continue = true;
            } else if (keyByte == 40) {
                instance.sender = ProtocolParser.readUInt64(stream);

                _continue = true;
            } else if (keyByte == 48) {
                instance.reliable = ProtocolParser.readBool(stream);

                _continue = true;
            } else if (keyByte == 114) {
                //PooledObjects.positionStreamPool:pop()
                if (instance.data == null) {
                    instance.data = RTData.readRTData(stream, instance.data);
                } else {
                    RTData.readRTData(stream, instance.data);
                }

                _continue = true;
            } else if (keyByte == 122) {
                instance.payload = instance.readPayload(stream, origLimit);

                _continue = true;
            }

            if (!_continue) {
                var key = ProtocolParser.readKey(keyByte, stream);
                if (key.field == 0) {
                    Logger.warn('Packet', 'Invalid field id: 0, something went wrong in the stream');

                    return 0;
                } else {
                    ProtocolParser.skipKey(stream, key);
                }
            }
        }

        return origLimit;
    };

    function assert(condition, message) {
        if (!condition) {
            message = message || 'Assertion failed';
            if (typeof Error !== 'undefined') {
                throw new Error(message);
            }
            throw message; // Fallback
        }
    }

    var Stream = function () {
        function Stream() {
            classCallCheck(this, Stream);

            this.buffer = '';
            this.position = 0;
        }

        createClass(Stream, [{
            key: 'getPosition',
            value: function getPosition() {
                return this.position;
            }
        }, {
            key: 'setPosition',
            value: function setPosition(position) {
                this.position = position;
            }
        }, {
            key: 'getLength',
            value: function getLength() {
                return this.buffer.length;
            }
        }, {
            key: 'bytesAvailable',
            value: function bytesAvailable() {
                return this.buffer.length - this.position;
            }
        }, {
            key: 'readByte',
            value: function readByte() {
                var ret = this.readChar();

                if (typeof ret === 'number' && ret == -1) {
                    return 0;
                } else {
                    return ret.charCodeAt(0);
                }
            }
        }, {
            key: 'readChar',
            value: function readChar() {
                var buffer = '';
                var totalBytes;
                var ret;

                ret = this.readChars(buffer, this.position, 1);
                totalBytes = ret[0];
                buffer = ret[1];
                if (totalBytes == 1) {
                    return buffer.slice(0, 1);
                } else {
                    return -1;
                }
            }
        }, {
            key: 'readChars',
            value: function readChars(buffer, position, length) {
                assert(typeof buffer === 'string', 'buffer must be string');

                if (this.bytesAvailable() <= 0) {
                    Logger.warn('Stream', 'Reached end of the stream');

                    return [0, buffer];
                }

                if (length <= 0) {
                    Logger.warn('Stream', 'no characters read (length = 0)');

                    return [0, buffer];
                }

                if (buffer.length > 0) {
                    assert(position >= 0 && position < buffer.length, 'position out of range');
                } else {
                    position = 0;
                }

                var subString;
                var newBuffer = '';
                var startPosition = this.position;
                var endPosition = startPosition + length;
                var newLength;

                if (endPosition > this.buffer.length) {
                    endPosition = this.buffer.length;
                }

                newLength = endPosition - startPosition;

                subString = this.buffer.slice(startPosition, endPosition);

                this.position = endPosition;

                if (position == 0) {
                    newBuffer = subString;

                    if (subString.length < buffer.length) {
                        newBuffer = newBuffer + buffer.slice(newLength);
                    }
                } else {
                    newBuffer = buffer.slice(0, position) + subString;

                    if (position + newLength + 1 <= buffer.length) {
                        newBuffer = newBuffer + buffer.slice(position + newLength);
                    }
                }

                return [newLength, newBuffer];
            }
        }, {
            key: 'read',
            value: function read(buffer, position, length) {
                assert((typeof buffer === 'undefined' ? 'undefined' : _typeof(buffer)) === 'object' && Array.isArray(buffer), 'buffer must be array');

                var i;
                var ret;
                var totalBytes;
                var newBuffer = '';
                var newArray = [];

                for (i = 0; i < buffer.length; i++) {
                    newBuffer = newBuffer + String.fromCharCode(buffer[i]);
                }

                ret = this.readChars(newBuffer, position, length);
                totalBytes = ret[0];
                newBuffer = ret[1];

                for (i = 0; i < newBuffer.length; i++) {
                    newArray.push(newBuffer.charCodeAt(i));
                }

                return [totalBytes, newArray];
            }
        }, {
            key: 'writeByte',
            value: function writeByte(byte) {
                assert(typeof byte === 'number', 'not valid byte');
                assert(byte >= 0 && byte <= 255, 'not valid byte');

                this.writeChar(String.fromCharCode(byte));
            }
        }, {
            key: 'writeChar',
            value: function writeChar(char) {
                this.writeChars(char, 0, 1);
            }
        }, {
            key: 'writeChars',
            value: function writeChars(buffer, position, length) {
                assert(typeof buffer === 'string', 'buffer must be string');

                assert(buffer && buffer.length > 0, 'buffer must not be nil or empty');

                if (this.bytesAvailable() < 0) {
                    Logger.warn('Stream', 'Reached end of the stream');

                    return;
                }

                if (length <= 0) {
                    Logger.warn('Stream', 'No characters written (length = 0)');

                    return [0, buffer];
                }

                assert(position >= 0 && position < buffer.length, 'position out of range');

                var subString;
                var newBuffer = '';
                var startPosition = position;
                var endPosition = startPosition + length;
                var newLength;

                if (endPosition > buffer.length) {
                    endPosition = buffer.length;
                }

                newLength = endPosition - startPosition;

                subString = buffer.slice(startPosition, endPosition);

                if (this.position == 0) {
                    newBuffer = subString;
                    if (this.buffer.length > subString.length) {
                        newBuffer = newBuffer + this.buffer.slice(newLength);
                    }
                } else {
                    newBuffer = this.buffer.slice(0, this.position) + subString;

                    if (this.position + newLength + 1 <= this.buffer.length) {
                        newBuffer = newBuffer + this.buffer.slice(this.position + newLength);
                    }
                }

                this.buffer = newBuffer;
                this.position = this.position + newLength;
            }
        }, {
            key: 'writeBytes',
            value: function writeBytes(buffer, position, length) {
                assert((typeof buffer === 'undefined' ? 'undefined' : _typeof(buffer)) === 'object' && Array.isArray(buffer), 'buffer must be array');

                var newBuffer = '';

                for (var i = position; i < position + length; i++) {
                    if (i >= buffer.length) {
                        break;
                    }

                    newBuffer = newBuffer + String.fromCharCode(buffer[i]);
                }

                this.writeChars(newBuffer, 0, length);
            }
        }, {
            key: 'seek',
            value: function seek(offset) {
                this.position = this.position - offset;

                if (this.position < 0) {
                    this.position = 0;
                } else if (this.position >= this.buffer.length) {
                    this.position = this.buffer.length - 1;
                }
            }
        }, {
            key: 'toHex',
            value: function toHex() {
                var ret = '';
                var value;
                var a;

                for (var i = 0; i < Math.ceil(this.buffer.length / 16) * 16; i++) {
                    if (i % 16 == 0) {
                        value = i.toString(16);

                        for (a = value.length; a < 8; a++) {
                            ret = ret + '0';
                        }
                        ret = ret + value + '  ';
                    }
                    if (i >= this.buffer.length) {
                        ret = ret + '## ';
                    } else {
                        value = this.buffer.charCodeAt(i).toString(16);

                        for (a = value.length; a < 2; a++) {
                            ret = ret + '0';
                        }
                        ret = ret + value + ' ';
                    }
                    if ((i + 1) % 8 == 0) {
                        ret = ret + ' ';
                    }
                    if ((i + 1) % 16 == 0) {
                        value = this.buffer.slice(i - 16 + 1, i + 1);

                        ret = ret + value.replace('[^\x20-\x7E]', '.') + '\n';
                    }
                }

                return ret;
            }
        }, {
            key: 'toString',
            value: function toString() {
                return this.buffer;
            }
        }, {
            key: 'getBuffer',
            value: function getBuffer() {
                var buffer = [];

                for (var i = 0; i < this.buffer.length; i++) {
                    buffer.push(this.buffer.charCodeAt(i));
                }

                return buffer;
            }
        }]);
        return Stream;
    }();

    var PooledObjects = {};

    PooledObjects.packetPool = new ObjectPool(function () {
        return new Packet();
    }, function (packet) {
        packet.reset();
    }, 5);

    PooledObjects.memoryStreamPool = new ObjectPool(function () {
        return new Stream();
    }, function (stream) {
        stream.setPosition(0);
    }, 5);

    PooledObjects.positionStreamPool = {};

    PooledObjects.limitedPositionStreamPool = new ObjectPool(function () {
        return new LimitedPositionStream();
    }, null, 5);

    PooledObjects.byteBufferPool = new ObjectPool(function () {
        var array = [];

        for (var i = 0; i < 256; i++) {
            array.push(0);
        }

        return array;
    }, null, 5);

    PooledObjects.customRequestPool = new ObjectPool(function () {
        return new CustomRequest();
    }, function (cr) {
        cr.reset();
    }, 5);

    var RTPacket = function () {
        function RTPacket(opCode, sender, limitedStream, limit, data, packetSize) {
            classCallCheck(this, RTPacket);

            this.opCode = opCode;
            this.sender = sender;
            this.stream = limitedStream;
            this.streamLength = limit;
            this.data = data;
            this.packetSize = packetSize;
        }

        createClass(RTPacket, [{
            key: 'toString',
            value: function toString() {
                var string = 'OpCode=' + this.opCode + ',Sender=' + this.sender + ',streamExists=';

                if (this.stream != null) {
                    string = string + 'true,StreamLength=' + this.streamLength;
                } else {
                    string = string + 'false';
                }

                string = string + ',Data=';

                if (this.data != null) {
                    string = string + this.data.toString();
                } else {
                    string = string + '.PacketSize=' + this.packetSize;
                }

                return string;
            }
        }]);
        return RTPacket;
    }();

    var CustomCommand = function () {
        function CustomCommand() {
            classCallCheck(this, CustomCommand);

            this.session = null;
            this.opCode = 0;
            this.sender = 0;
            this.limit = 0;
            this.packetSize = 0;
            this.ms = null;
            this.limitedStream = null;
            this.data = null;
        }

        createClass(CustomCommand, [{
            key: 'configure',
            value: function configure(opCode, sender, lps, data, limit, session, packetSize) {
                this.ms = PooledObjects.memoryStreamPool.pop();
                this.packetSize = packetSize;
                this.opCode = opCode;
                this.sender = sender;
                this.data = data;
                this.session = session;
                this.limit = limit;
                this.limitedStream = null;

                if (lps != null) {
                    this.limitedStream = PooledObjects.limitedPositionStreamPool.pop();

                    for (var i = 1; i <= limit; i++) {
                        var read = lps.readByte();

                        this.ms.writeByte(read);
                    }
                    this.ms.setPosition(0);
                    this.limitedStream.wrap(this.ms, limit);
                }

                return this;
            }
        }, {
            key: 'execute',
            value: function execute() {
                this.session.onPacket(new RTPacket(this.opCode, this.sender, this.limitedStream, this.limit, this.data, this.packetSize));

                PooledObjects.memoryStreamPool.push(this.ms);
                PooledObjects.limitedPositionStreamPool.push(this.limitedStream);
                CustomCommand.pool.push(this);
            }
        }]);
        return CustomCommand;
    }();

    CustomCommand.pool = new ObjectPool(function () {
        return new CustomCommand();
    }, null, 5);

    var Connection = function () {
        function Connection(session) {
            classCallCheck(this, Connection);

            this.session = session;
            this.stopped = false;
        }

        createClass(Connection, [{
            key: 'stop',
            value: function stop() {
                this.stopped = true;
                this.stopInternal();
            }
        }, {
            key: 'onPacketReceived',
            value: function onPacketReceived(p, packetSize) {
                if (p.command) {
                    if (p.command['abstractResultType']) {
                        var result = p.command;

                        result.configure(p, this.session);
                        if (result.executeAsync()) {
                            this.session.submitAction(p.command);
                        } else {
                            p.command.execute();
                        }
                    } else {
                        this.session.submitAction(p.command);
                    }
                } else {
                    if (!p.hasPayload) {
                        var cmd;

                        if (p.sender) {
                            cmd = CustomCommand.pool.pop().configure(p.opCode, p.sender, null, p.data, 0, this.session, packetSize);
                        } else {
                            cmd = CustomCommand.pool.pop().configure(p.opCode, 0, null, p.data, 0, this.session, packetSize);
                        }

                        if (cmd) {
                            this.session.submitAction(cmd);
                        }
                    }
                }
            }
        }]);
        return Connection;
    }();

    var LoginCommand = function (_RTRequest) {
        inherits(LoginCommand, _RTRequest);

        function LoginCommand(token) {
            classCallCheck(this, LoginCommand);

            var _this = possibleConstructorReturn(this, (LoginCommand.__proto__ || Object.getPrototypeOf(LoginCommand)).call(this));

            _this.opCode = 0;
            _this.token = token;
            _this.clientVersion = 2;
            return _this;
        }

        createClass(LoginCommand, [{
            key: 'toPacket',
            value: function toPacket(session, fast) {
                var p = PooledObjects.packetPool.pop();

                p.opCode = this.opCode;
                p.data = this.data;
                p.session = session;

                if (!fast && this.intent != Constants.deliveryIntent.RELIABLE) {
                    p.reliable = false;
                }

                if (this.intent == Constants.deliveryIntent.UNRELIABLE_SEQUENCED) {
                    p.sequenceNumber = session.nextSequenceNumber();
                }

                if (this.targetPlayers.length > 0) {
                    p.targetPlayers = this.targetPlayers;
                }

                p.request = this;

                return p;
            }
        }, {
            key: 'serialize',
            value: function serialize(stream) {
                if (this.token != null) {
                    stream.writeByte(10);
                    ProtocolParser.writeString(stream, this.token);
                }
                if (this.clientVersion != null) {
                    stream.writeByte(16);
                    ProtocolParser.writeUInt64(stream, this.clientVersion);
                }
            }
        }]);
        return LoginCommand;
    }(RTRequest);

    var ReliableConnection = function (_Connection) {
        inherits(ReliableConnection, _Connection);

        function ReliableConnection(remotehost, remoteport, session) {
            classCallCheck(this, ReliableConnection);

            var _this = possibleConstructorReturn(this, (ReliableConnection.__proto__ || Object.getPrototypeOf(ReliableConnection)).call(this, session));

            _this.remotehost = remotehost;
            _this.remoteport = remoteport;
            return _this;
        }

        createClass(ReliableConnection, [{
            key: 'connect',
            value: function connect() {
                // TODO: revert back to 'wss://' once we support secure connections
                this.client = new WebSocket('ws://' + this.remotehost + ':' + this.remoteport);
                this.client.binaryType = 'arraybuffer';
                this.client.onopen = this.onopen.bind(this);
                this.client.onclose = this.onclose.bind(this);
                this.client.onerror = this.onerror.bind(this);
                this.client.onmessage = this.onmessage.bind(this);
            }
        }, {
            key: 'onopen',
            value: function onopen() {
                if (this.stopped || this.session == null) {
                    if (this.client != null) {
                        this.client.close();
                        this.client = null;
                    }

                    return;
                }

                this.session.log('ReliableConnection', Logger.level.DEBUG, 'TCP Connection Established');

                var loginCmd = new LoginCommand(this.session.getSignedToken(this.remotehost, this.remoteport));

                this.send(loginCmd);
            }
        }, {
            key: 'onmessage',
            value: function onmessage(e) {
                if (this.stopped) {
                    if (this.client != null) {
                        this.client.close();
                        this.client = null;
                    }

                    return;
                }

                var data = e.data;
                var read = data.byteLength;
                var rss = new Stream();
                var array = Array.from(new Uint8Array(data));

                rss.writeBytes(array, 0, read);

                rss.setPosition(0);
                while (rss.getPosition() < read) {
                    var p = PooledObjects.packetPool.pop();
                    var bytesRead = this.read(rss, p);

                    this.onPacketReceived(p, bytesRead);
                    PooledObjects.packetPool.push(p);
                    p = PooledObjects.packetPool.pop();
                }
            }
        },  {
            key: 'onclose',
            value: function onclose() {
                var _this2 = this;

                if (this.session != null) {
                    this.session.log('ReliableConnection', Logger.level.DEBUG, 'TCP Connection Closed');
                }

                if (!this.stopped && this.client != null) {
                    // Attempt a reconnection after a delay.
                    setTimeout(function () {
                        _this2.session.log('ReliableConnection', Logger.level.DEBUG, 'Reconnecting');
                        _this2.connect();
                    }, 1000);
                }
            }
        }, {
            key: 'onerror',
            value: function onerror(e) {
                if (this.session != null) {
                    this.session.log('ReliableConnection', Logger.level.DEBUG, 'TCP Connection Error: ' + e.message);
                }
            }
        }, {
            key: 'send',
            value: function send(request) {
                if (this.client != null && this.client.readyState == WebSocket.OPEN) {
                    var stream = new Stream();
                    var p = request.toPacket(this.session, false);
                    var ret = Packet.serializeLengthDelimited(stream, p);
                    var buffer = new Uint8Array(stream.getBuffer());

                    this.client.send(buffer);

                    PooledObjects.packetPool.push(p);

                    return ret;
                } else {
                    return -1;
                }
            }
        }, {
            key: 'read',
            value: function read(stream, p) {
                if (this.stopped) {
                    return 0;
                }

                p.session = this.session;
                p.reliable = true;

                var ret = Packet.deserializeLengthDelimited(stream, p);

                if (p.reliable == null) {
                    p.reliable = true;
                }

                return ret;
            }
        }, {
            key: 'stopInternal',
            value: function stopInternal() {
                if (this.client != null && this.client.readyState == WebSocket.OPEN) {
                    this.client.close();
                    this.client = null;
                }
            }
        }]);
        return ReliableConnection;
    }(Connection);

    var RTSessionImpl = function () {
        function RTSessionImpl(gamesparks, connectToken, hostName, tcpPort, listener) {
            classCallCheck(this, RTSessionImpl);

            this.gamesparks = gamesparks;
            this.connectToken = connectToken;
            this.tcpPort = tcpPort;
            this.fastPort = tcpPort;
            this.hostName = hostName;
            this.activePeers = [];
            this.running = false;
            this.ready = false;
            this.actionQueue = [];
            this.connectState = Constants.connectState.DISCONNECTED;
            this.mustConnnectBy = new Date().getTime();
            this.reliableConnection = null;
            this.sequenceNumber = 0;
            this.peerId = null;
            this.peerMaxSequenceNumbers = [];
            this.listener = listener;
        }

        createClass(RTSessionImpl, [{
            key: 'start',
            value: function start() {
                this.running = true;
            }
        }, {
            key: 'stop',
            value: function stop() {
                this.log('IRTSession', Logger.level.DEBUG, 'Stopped');

                this.running = false;
                this.ready = false;

                if (this.reliableConnection) {
                    this.reliableConnection.stop();
                }

                this.setConnectState(Constants.connectState.DISCONNECTED);
            }
        }, {
            key: 'update',
            value: function update() {
                if (this.running) {
                    this.checkConnection();
                }

                var toExecute = null;

                for (;;) {
                    toExecute = this.getNextAction();
                    if (toExecute) {
                        toExecute.execute();
                    } else {
                        break;
                    }
                }
            }
        }, {
            key: 'getNextAction',
            value: function getNextAction() {
                if (this.actionQueue.length > 0) {
                    return this.actionQueue.shift();
                }

                return null;
            }
        }, {
            key: 'submitAction',
            value: function submitAction(action) {
                this.actionQueue.push(action);
            }
        }, {
            key: 'checkConnection',
            value: function checkConnection() {
                if (this.connectState == Constants.connectState.DISCONNECTED) {
                    this.log('IRTSession', Logger.level.INFO, 'Disconnected, trying to connect');

                    this.setConnectState(Constants.connectState.CONNECTING);

                    this.connectReliable();
                } else if (this.connectState == Constants.connectState.CONNECTING && new Date().getTime() > this.mustConnnectBy) {
                    this.setConnectState(Constants.connectState.DISCONNECTED);

                    this.log('IRTSession', Logger.level.INFO, 'Not connected in time, retrying');

                    if (this.reliableConnection) {
                        this.reliableConnection.stopInternal();
                        this.reliableConnection = null;
                    }
                }
            }
        }, {
            key: 'setConnectState',
            value: function setConnectState(value) {
                if (value == Constants.connectState.RELIABLE_AND_FAST_SEND || value == Constants.connectState.RELIABLE_AND_FAST) {
                    value = Constants.connectState.RELIABLE_ONLY;
                }

                if (value != this.connectState) {
                    if (this.connectState < value || value == Constants.connectState.DISCONNECTED) {
                        this.log('IRTSession', Logger.level.DEBUG, 'State Change : from ' + this.connectState + ' to ' + value + ', ActivePeers ' + this.activePeers.length);

                        this.connectState = value;

                        if (value == Constants.connectState.RELIABLE_ONLY) {
                            this.onReady(true);
                        }
                    }
                }
            }
        }, {
            key: 'connectReliable',
            value: function connectReliable() {
                this.mustConnnectBy = new Date().getTime() + Constants.TCP_CONNECT_TIMEOUT_SECONDS * 1000;

                this.reliableConnection = new ReliableConnection(this.hostName, this.tcpPort, this);
                this.reliableConnection.connect();
            }
        }, {
            key: 'nextSequenceNumber',
            value: function nextSequenceNumber() {
                var sequenceNumber = this.sequenceNumber;

                this.sequenceNumber = this.sequenceNumber + 1;

                return sequenceNumber;
            }
        }, {
            key: 'shouldExecute',
            value: function shouldExecute(peerId, sequence) {
                if (sequence == null) {
                    return true;
                } else if (this.peerMaxSequenceNumbers[peerId] == null) {
                    this.peerMaxSequenceNumbers[peerId] = 0;
                }

                if (this.peerMaxSequenceNumbers[peerId] > sequence) {
                    this.log('IRTSession', Logger.level.DEBUG, 'Discarding sequence id ' + sequence + ' from peer ' + peerId.toString());

                    return false;
                } else {
                    this.peerMaxSequenceNumbers[peerId] = sequence;

                    return true;
                }
            }
        }, {
            key: 'resetSequenceForPeer',
            value: function resetSequenceForPeer(peerId) {
                if (this.peerMaxSequenceNumbers[peerId]) {
                    this.peerMaxSequenceNumbers[peerId] = 0;
                }
            }
        }, {
            key: 'onPlayerConnect',
            value: function onPlayerConnect(peerId) {
                this.resetSequenceForPeer(peerId);
                if (this.listener) {
                    if (this.ready) {
                        this.listener.onPlayerConnect(peerId);
                    }
                }
            }
        }, {
            key: 'onPlayerDisconnect',
            value: function onPlayerDisconnect(peerId) {
                if (this.listener) {
                    if (this.ready) {
                        this.listener.onPlayerDisconnect(peerId);
                    }
                }
            }
        }, {
            key: 'onReady',
            value: function onReady(ready) {
                if (!this.ready && ready) {
                    this.sendData(OpCodes.PLAYER_READY_MESSAGE, Constants.deliveryIntent.RELIABLE, null, null);
                    if (this.peerId) {
                        var ok = false;

                        for (var k in this.activePeers) {
                            if (this.activePeers[k] == this.peerId) {
                                ok = true;

                                break;
                            }
                        }

                        if (!ok) {
                            this.activePeers.push(this.peerId);
                        }
                    }
                }

                this.ready = ready;

                if (!this.ready) {
                    this.setConnectState(Constants.connectState.DISCONNECTED);
                }

                if (this.listener) {
                    var myListener = this.listener;

                    this.submitAction(ActionCommand.pool.pop().configure(function () {
                        myListener.onReady(ready);
                    }));
                }
            }
        }, {
            key: 'onPacket',
            value: function onPacket(packet) {
                if (this.listener) {
                    this.listener.onPacket(packet);
                } else {
                    throw new Error('AccessViolationException');
                }
            }
        }, {
            key: 'sendData',
            value: function sendData(opCode, intent, payload, data, targetPlayers) {
                return this.sendRTDataAndBytes(opCode, intent, payload, data, targetPlayers);
            }
        }, {
            key: 'sendRTData',
            value: function sendRTData(opCode, intent, data, targetPlayers) {
                return this.sendRTDataAndBytes(opCode, intent, null, data, targetPlayers);
            }
        }, {
            key: 'sendBytes',
            value: function sendBytes(opCode, intent, payload, targetPlayers) {
                return this.sendRTDataAndBytes(opCode, intent, payload, null, targetPlayers);
            }
        }, {
            key: 'sendRTDataAndBytes',
            value: function sendRTDataAndBytes(opCode, intent, payload, data, targetPlayers) {
                var csr = PooledObjects.customRequestPool.pop();
                var ret;

                csr.configure(opCode, intent, payload, data, targetPlayers);
                if (this.connectState >= Constants.connectState.RELIABLE_ONLY) {
                    ret = this.reliableConnection.send(csr);

                    PooledObjects.customRequestPool.push(csr);

                    return ret;
                }

                PooledObjects.customRequestPool.push(csr);

                return 0;
            }
        }, {
            key: 'getSignedToken',
            value: function getSignedToken(host, port) {
                return this.gamesparks.signRealtimeToken(host, port, this.connectToken);
            }
        }, {
            key: 'log',
            value: function log(tag, level, msg) {
                if (Logger.shouldLog(tag, level)) {
                    this.submitAction(LogCommand.pool.pop().configure(this, tag, level, msg));
                }
            }
        }]);
        return RTSessionImpl;
    }();

    // Start with the constants.
    var GameSparksRT = Object.assign({}, Constants);

    // Expose logger levels and methods.

    GameSparksRT.logLevel = Logger.level;

    GameSparksRT.setRootLogLevel = function (level) {
        Logger.setRootLogLevel(level);
    };

    GameSparksRT.setLogLevel = function (tag, level) {
        Logger.setLogLevel(tag, level);
    };

    /**
     * Returns a realtime session.
     */
    GameSparksRT.getSession = function (gamesparks, connectToken, hostName, tcpPort, listener) {
        return new RTSessionImpl(gamesparks, connectToken, hostName, tcpPort, listener);
    };

    // Include the RTData and RTVector classes.
    GameSparksRT.RTData = RTData;
    GameSparksRT.RTVector = RTVector;

    // Include the onMessageReturn function.
    GameSparksRT.onMessageReturn = (e) => {
        var data = new Uint8Array(e);
        var read = data.byteLength;
        var rss = new Stream();
        var array = Array.from(new Uint8Array(data));

        rss.writeBytes(array, 0, read);

        rss.setPosition(0);

        const readBytes = (stream, p) => {
            p.reliable = true;

            var ret = Packet.deserializeLengthDelimited(stream, p);

            if (p.reliable == null) {
                p.reliable = true;
            }

            return ret;
        };

        var packets = [];
        while (rss.getPosition() < read) {
            var p = PooledObjects.packetPool.pop();
            readBytes(rss, p);

            let clone = Object.assign( Object.create( Object.getPrototypeOf(p)), p);
            packets.push(clone);
            PooledObjects.packetPool.push(p);
            p = PooledObjects.packetPool.pop();
        }
        return packets;
    };

    GameSparksRT.getMessageFromData = (opCode, intent, data, targetPlayers) => {
        var csr = PooledObjects.customRequestPool.pop();

        csr.configure(opCode, intent, null, data, targetPlayers);
        const send = (request) => {
            var stream = new Stream();
            var p = request.toPacket(null, false);
            Packet.serializeLengthDelimited(stream, p);
            var buffer = new Uint8Array(stream.getBuffer());

            PooledObjects.packetPool.push(p);

            return Buffer.from(buffer);
        };

        var ret = send(csr);

        PooledObjects.customRequestPool.push(csr);

        return ret;
    };

    GameSparksRT.serializeLengthDelimited = (stream, RTValue) => {
        var ms = PooledObjects.memoryStreamPool.pop();

        if (RTValue.string_val) {
            ms.writeByte(10);
            ProtocolParser.writeString(ms, RTValue.string_val);
        } else if (RTValue.data_val) {
            ms.writeByte(114);
            RTData.writeRTData(ms, this.data_val);
        } else if (RTValue.vec_val) {
            var vec_value = this.vec_val;
            var numberOfFloatsSet = 0;

            ms.writeByte(18);

            if (vec_value.x != null) {
                numberOfFloatsSet = numberOfFloatsSet + 1;
            }
            if (vec_value.y != null) {
                numberOfFloatsSet = numberOfFloatsSet + 1;
            }
            if (vec_value.z != null) {
                numberOfFloatsSet = numberOfFloatsSet + 1;
            }
            if (vec_value.w != null) {
                numberOfFloatsSet = numberOfFloatsSet + 1;
            }

            ProtocolParser.writeUInt32(ms, 4 * numberOfFloatsSet);

            for (var i = 1; i <= numberOfFloatsSet; i++) {
                if (i == 1) {
                    ProtocolParser.writeSingle(ms, vec_value.x);
                } else if (i == 2) {
                    ProtocolParser.writeSingle(ms, vec_value.y);
                } else if (i == 3) {
                    ProtocolParser.writeSingle(ms, vec_value.z);
                } else if (i == 4) {
                    ProtocolParser.writeSingle(ms, vec_value.w);
                }
            }
        }

        var data = ms.getBuffer();

        ProtocolParser.writeBytes(stream, data, ms.getPosition());

        PooledObjects.memoryStreamPool.push(ms);
    };

    return GameSparksRT;

})));
