#!/usr/bin/env node

'use strict';

const fs = require('fs');
const path = require('path');


function Stringize(object) {
    let result = "{";
    let separator = "";
    for (let prop in object) {
        if (object.hasOwnProperty(prop)) {
            result += separator + prop + "=" + object[prop];
            separator = ",";
        }
    }
    result += "}";
    return result;
}

function isValidPortRange(interval) {
    return (
        (interval != null) &&
        Number.isInteger(interval.start) &&
        Number.isInteger(interval.end) &&
        (interval.start > 0) &&
        (interval.end >= interval.start) &&
        (interval.end <= 65534)
    );
}

function intervalsOverlap(interval1, interval2) {
    if (interval1.start < interval2.start) {
        return interval1.end >= interval2.start;
    } else {
        return interval2.end >= interval1.start;
    }
}

function parseCommandLine() {
    const DEFAULT_SERVER_SCRIPT = "app/server.js";

    const ArgumentParser = require('argparse').ArgumentParser;
    let parser = new ArgumentParser({
        prog: 'phantom',
        addHelp: true
    });
    parser.addArgument(
        ['-t', '--test-dist'], {
            help: 'Test app prior to bundling',
            action: 'storeTrue',
            defaultValue: false
        }
    );
    parser.addArgument(
        ['-c', '--client-enabled'], {
            help: 'Enable client test port for debugging',
            action: 'storeTrue',
            defaultValue: false
        }
    );
    parser.addArgument(
        ['-r', '--client-required'], {
            help: 'Exit if client test port cannot be opened',
            action: 'storeTrue',
            defaultValue: false
        }
    );
    parser.addArgument(
        ['-s', '--script'], {
            help: `Set server script (default: ${DEFAULT_SERVER_SCRIPT})`,
            defaultValue: DEFAULT_SERVER_SCRIPT
        }
    );
    parser.addArgument(
        ['-p', '--port'], {
            help: 'Set starting port number',
            type: 'int',
            defaultValue: 3001
        }
    );
    parser.addArgument(
        ['-e', '--end'], {
            help: 'Set ending port number',
            metavar: 'PORT',
            type: 'int',
            defaultValue: null
        }
    );

    return parser.parseArgs();
}

function queryOpenClient(options) {
    return options.client_enabled || options.client_required || options.test_dist;
}

function getServerPorts(options) {
    let ports = {};
    ports.clientTestPortRange = null;
    ports.gameServerPortRange = {};
    ports.gameServerPortRange.start = options.port;
    ports.gameServerPortRange.end = options.end || options.port;

    // Validate ports
    if (!isValidPortRange(ports.gameServerPortRange)) {
        throw new Error(`Invalid game server port range: ${Stringize(ports.gameServerPortRange)}`);
    }

    // If the client ports are enabled then they must not overlap the game server ports!
    if (queryOpenClient(options)) {
        ports.clientTestPortRange = {};
        ports.clientTestPortRange.start = 3000;
        ports.clientTestPortRange.end = ports.clientTestPortRange.start + ports.gameServerPortRange.end - ports.gameServerPortRange.start;
        if (!isValidPortRange(ports.clientTestPortRange)) {
            throw new Error(`Invalid client test port range: ${Stringize(ports.clientTestPortRange)}`);
        }
        if (intervalsOverlap(ports.gameServerPortRange, ports.clientTestPortRange)) {
            throw new Error(`Overlapping port ranges: gameServer(${Stringize(ports.gameServerPortRange)}) clientTest:(${Stringize(ports.clientTestPortRange)})`);
        }
    }

    return ports;
}

async function openPortFromRangeAsync(interval, asyncCallback) {
    for (let port = interval.start; port <= interval.end; ++port) {
        try {
            return await asyncCallback(port);
        } catch (err) {
            // The port is probably in use and we're getting EACCESS or EADDRINUSE;
            // ignore and keep searching for an open port in the interval
        }
    }
    throw new Error("port not found");
}

function loadDeveloperCode(script) {
    // No script specified?
    if (!script) {
        return {};
    }

    //  Load developer game-server code
    const ssPath = path.join(process.cwd(), script);
    if (!fs.existsSync(ssPath)) {
        throw new Error(`script ${script} not found`);
    }

    console.log(`Loading game server script '${ssPath}'`);
    return require(ssPath).ssExports;
}

exports.GameServerRunner = class GameServerRunner {
    constructor() {
        this.options = parseCommandLine();
        this.ports = getServerPorts(this.options);
    }

    async _initGameServer(ssExports) {
        // Create game server
        const Server = require('./server/index.js').Server;
        const connector = require('./server/connectors/ws.js');
        const configuration = Object.assign(
            require('./default-configuration.js').defaultConfiguration,
            ssExports.configuration
        );

        // Start game server
        const server = new Server(configuration, ssExports, connector.connectorExports);
        const gameServer = new connector.WSConnector(server);
        return openPortFromRangeAsync(
            this.ports.gameServerPortRange,
            (port) => gameServer.start(port, ssExports.init)
        ).then(port => {
            console.log(`Game server port: ${port}`);
            return port;
        }).catch(err => {
            console.log(`ERROR: failed to open game server port: ${err}`);
            process.exit(1);
        });
    }

    async _initClientTest(gameServerPort) {
        if (!queryOpenClient(this.options)) {
            return Promise.resolve();
        }

        // Start an http server for web browsers for testing
        // Now that we know which port is used for the game server, pick clientTestPort so that it matches
        // e.g. gameServerPortRange => { 4000 - 4999 }, clientTestPortRange = { 3000, 3999 }
        //      actualGameServerPort = 4123 ==> let actualClientTestPort = 3123.
        let clientTestPortRange = {};
        clientTestPortRange.start = gameServerPort - this.ports.gameServerPortRange.start + this.ports.clientTestPortRange.start;
        clientTestPortRange.end = clientTestPortRange.start;

        const tester = require('./tester');
        const testerServer = new tester.Server();
        return openPortFromRangeAsync(
            clientTestPortRange,
            (port) => {
                testerServer.start(port);
                return port;
            }
        ).then(port => {
            console.log(`Client test port: ${port}`);
        }).catch(err => {
            if (this.options.client_required) {
                console.error(`ERROR: failed to open test client port: ${err}`);
                process.exit(1);
            } else {
                console.warn(`WARN: failed to open test client port: ${err}`);
            }
        });
    }

    async run_async() {
        let ssExports = loadDeveloperCode(this.options.script);
        return this._initGameServer(ssExports)
            .then(gameServerPort => {
                this._initClientTest(gameServerPort);
                return gameServerPort;
            })
            .then(gameServerPort => {
                if (this.options.test_dist) {
                    console.log("Success; exiting per '--test-run' command-line option");
                    process.exit(0);
                }
                return gameServerPort;
            })
            .catch(err => {
                console.error(`ERROR: unknown failure: ${err}`);
                process.exit(1);
            });
    }

    OnStartGameSession (session, callback) { // eslint-disable-line no-unused-vars
        // TODO: do something with session
        callback();
    }

    OnUpdateGameSession (session) { // eslint-disable-line no-unused-vars
        // TODO: do something with session
    }

    OnProcessTerminate (callback) {
        // TODO: alert users; call callback only when ready to terminate
        callback();
    }
};

// If this is the main application then load and run the game
if (typeof require != 'undefined' && require.main == module) {
    let gameServerRunner = new exports.GameServerRunner();
    gameServerRunner.run_async();
}
