const RTSession = require('../model/rt-session.js').RTSession;

const http = require('http');
const WebSocket = require('ws');

exports.WSConnector = class {
    // Setup the WebSocket server
    constructor(server) {
        // Server class
        this.server = server;
        server.startHeartbeat();

        this.socketsNotValidatedYet = new Array(2048); // 2^^11, should be updated to match real world, customer needs, and stress tests

        // Servers
        this.httpConnector = new http.createServer();
        this.wsConnector = new WebSocket.Server({
            server: this.httpConnector
        });

        // Server listeners
        this.wsConnector.on('connection', (ws, req) => {
            // Validate connection
            if (!this.server.canAcceptConnection()) {
                ws.terminate();
                return;
            }

            // TODO: add async GameLift validation. Once validated call below function as a callback or promise
            // Also add DDOS protection, using the socketsNotValidatedYet array
            this.connect(ws, req);
        });
    }

    connect(ws, req) {
        const peerId = this.server.onOpen(ws);
        if (!peerId) {
            return ws.terminate();
        }
        console.log(`Player ${peerId} connecting from ${req.connection.remoteAddress}:${req.connection.remotePort}`);

        // Set up websocket listeners
        ws.on('message', (message) => {
            this.server.onMessage(peerId, message);
        });
        ws.on('close', () => {
            this.server.onClose(peerId);
        });
        ws.on("pong", () => {
            this.server.onPong(peerId);
        });
    }

    // Start the WebSocket server
    start(port, init) {
        return new Promise((resolve, reject) => {
            this.httpConnector.listen(port, () => {
                if(init) {
                    init(new RTSession(this.server));
                }
                return resolve(port);
            });

            // NOTE: I learned the hard way that you must use wsConnector, not httpConnector, otherwise
            // errors are handled by the default wsConnector error handler prior to this code!
            this.wsConnector.on('error', err => reject(err));
        });
    }
};

exports.connectorExports = {
    sendMessageToClient: (ws, message) => {
        if(ws.readyState === ws.OPEN) {
            ws.send(message);
        }
    },
    terminateClient: (ws) => {
        ws.terminate();
    },
    pingClient: (ws) => {
        if(ws.readyState === ws.OPEN) {
            ws.ping();
        }
    }
};
