const RTPlayerContainer = require("./helpers/rt-player-container.js").RTPlayerContainer;

const CoreCommands_pb = require('../proto/CoreCommands_pb');
const Packet_pb = require('../proto/Packet_pb');

const gsrt = require('../gamesparks/gamesparks.rt.js');

exports.Server = class {
    // Setup the GameSparks server
    constructor(configuration, ssExports, connectorExports) {
        // Configuration
        this.maxPlayers = configuration.maxPlayers;
        this.maxMessageSize = configuration.maxMessageSize;
        this.pingIntervalTime = configuration.pingIntervalTime;

        const noOp = () => undefined;

        // Callbacks
        const onPacketDefault = (packet) => {return packet;};
        this.onPacket = ssExports.onPacket || onPacketDefault;

        const onPlayerConnectDefault = () => {return true;};
        this.onPlayerConnect = ssExports.onPlayerConnect || onPlayerConnectDefault;

        const onPlayerDisconnectDefault = noOp;
        this.onPlayerDisconnect = ssExports.onPlayerDisconnect || onPlayerDisconnectDefault;

        // Connector specific methods
        this.sendMessageToClient = connectorExports.sendMessageToClient || noOp;
        this.terminateClient = connectorExports.terminateClient || noOp;
        this.pingClient = connectorExports.pingClient || noOp;

        // RTPlayer
        this.RTPlayerContainer = new RTPlayerContainer(this.maxPlayers);
    }

    startHeartbeat() {
        // Initialize client/server heartbeat
        setInterval(() => this.pingPlayers(), this.pingIntervalTime);
    }

    pingPlayers() {
        const players = this.RTPlayerContainer.getPlayers();
        players.forEach((player) => {
            if(player.pinged) {
                this.terminateClient(player.client);
            } else {
                player.pinged = true;
                this.pingClient(player.client);
            }
        });
    }

    onPong(peerId) {
        this.RTPlayerContainer.getPlayers().get(peerId).pinged = false;
    }

    canAcceptConnection() {
        // Implement validation {GLIFT-9158}
        return this.RTPlayerContainer.getNumberOfPlayers() < this.maxPlayers;
    }

    // Creates and adds new player, returns peer ID of new player, 0 if doesn't connect
    addRTPlayer(client) {
        return this.RTPlayerContainer.createPlayer(client, this.onPlayerConnect);
    }

    // Removes player with specified peer ID
    removeRTPlayer(peerId) {
        this.onPlayerDisconnect(this.RTPlayerContainer.getPlayers()[peerId]);
        this.RTPlayerContainer.removePlayer(peerId);
    }

    terminatePlayerByPeerId(peerId) {
        this.terminateClient(this.RTPlayerContainer.getPlayers()[peerId].client);
    }

    onMessage(peerId, message) {
        try {
            if(message.length > this.maxMessageSize) {
                this.terminatePlayerByPeerId(peerId);
                throw new Error("Player " + peerId + " terminated. Sent too long of a message.");
            }
            const size = message.readInt8() + 1;
            if (size === message.length) {
                const input = Packet_pb.Packet.deserializeBinary(message.slice(1)).toObject();
                switch (input.opcode) {
                // Player Ready
                case -7: {
                    console.log(`Player ${peerId} ready`);
                    break;
                }
                // Login Command
                case 0: {
                    const loginCommand = CoreCommands_pb.LoginCommand.deserializeBinary(input.payload).toObject();
                    console.log(`Player ${peerId} logging in`);

                    const loginResult = new CoreCommands_pb.LoginResult();
                    loginResult.setSuccess(true);
                    loginResult.setPeerid(peerId);
                    loginResult.setActivepeersList(this.RTPlayerContainer.getPlayers());
                    loginResult.setReconnecttoken(loginCommand.token);
                    const loginResultData = loginResult.serializeBinary();

                    const output = new Packet_pb.Packet();
                    output.setOpcode(-1);
                    output.setPayload(loginResultData);
                    const outputData = output.serializeBinary();

                    const players = this.RTPlayerContainer.getPlayers();
                    this.sendMessageToClient(players.get(peerId).client, this.packageData(outputData));
                    break;
                }
                default: {
                    // TODO: handle incoming packets (validate, run customer callback, etc.)
                    console.log(`Player ${peerId} sent a ${size} byte packet with opcode '${input.opcode}'`);

                    // TODO: Hook this up to the packet sending. This returns an array of all packets sent
                    const packets = gsrt.onMessageReturn(message);
                    if(packets) {
                        for(var packet of packets) {
                            // On Packet callback
                            const packetToSend = this.onPacket(packet);

                            if(packetToSend) {
                                const targetPlayers = input.targetplayersList;
                                if(targetPlayers.length == 0) {
                                    this.broadcastToAllExceptSender(peerId, message);
                                } else {
                                    this.broadcastToPeerIds(targetPlayers, message);
                                }
                            }
                        }
                    }
                }
                }
            } else {
                // TODO: handle multiple packets, or a partial packet, per frame
                console.log(`Player ${peerId} sent ${message.length} bytes when we expected ${size} bytes`);
            }
        } catch (e) {
            console.error(e);
        }
    }

    onOpen(client) {
        const peerId = this.addRTPlayer(client);
        if(!peerId) {
            return null;
        }

        console.log(`Player ${peerId} connected`);
        const playerConnect = new CoreCommands_pb.PlayerConnectMessage();
        playerConnect.setPeerid(peerId);
        playerConnect.setActivepeersList(this.RTPlayerContainer.getPlayers());
        const playerConnectData = playerConnect.serializeBinary();

        const output = new Packet_pb.Packet();
        output.setOpcode(-101);
        output.setPayload(playerConnectData);
        const outputData = output.serializeBinary();

        this.broadcastToAllExceptSender(peerId, this.packageData(outputData));
        return peerId;
    }

    onClose(peerId) {
        console.log(`Player ${peerId} disconnected`);
        this.removeRTPlayer(peerId);

        const playerDisconnect = new CoreCommands_pb.PlayerDisconnectMessage();
        playerDisconnect.setPeerid(peerId);
        playerDisconnect.setActivepeersList(this.RTPlayerContainer.getPlayers());
        const playerDisconnectData = playerDisconnect.serializeBinary();

        const output = new Packet_pb.Packet();
        output.setOpcode(-103);
        output.setPayload(playerDisconnectData);
        const outputData = output.serializeBinary();

        this.broadcastToAllExceptSender(peerId, this.packageData(outputData));
    }

    // Takes in a shouldSend function that determines if message should be sent to each peerId
    broadcast(shouldSend, message) {
        this.RTPlayerContainer.getPlayers().forEach((player, peerId) => {
            if(shouldSend(peerId)) {
                this.sendMessageToClient(player.client, message);
            }
        });
    }

    broadcastToAllExceptSender(senderPeerId, message) {
        const shouldSend = (peerId) => peerId !== senderPeerId;
        this.broadcast(shouldSend, message);
    }

    broadcastToPeerIds(peerIds, message) {
        const shouldSend = (peerId) => peerIds.includes(peerId);
        this.broadcast(shouldSend, message);
    }

    packageData(data) {
        return Buffer.concat([new Uint8Array([data.length]), data], data.length + 1);
    }
};
