package main

import (
	"github.com/cactus/go-statsd-client/statsd"

	"time"

	"code.justin.tv/foundation/twitchserver"
	"code.justin.tv/samus/nitro/config"
	"code.justin.tv/samus/nitro/internal/api"
	"code.justin.tv/samus/nitro/internal/app/grandfathering"
	"code.justin.tv/samus/nitro/internal/app/premium"
	"code.justin.tv/samus/nitro/internal/clients"
	"code.justin.tv/samus/nitro/metrics"
	log "github.com/sirupsen/logrus"
)

const (
	prodEnvironment = "prod"
)

func main() {
	environment := config.GetEnvironment()

	logLevel := log.DebugLevel
	if environment == prodEnvironment {
		logLevel = log.InfoLevel
	}
	log.SetLevel(logLevel)
	log.SetFormatter(&log.TextFormatter{FullTimestamp: true, TimestampFormat: time.RFC3339})

	stats, err := statsd.NewNoopClient()
	if err != nil {
		log.WithError(err).Fatal("Failed to initialize statsd client")
	}
	err = startWebServer(stats)
	if err != nil {
		log.Fatalf("Error during server shutdown: %v", err)
	}
}

// Set up the web server start listening for requests.
func startWebServer(stats statsd.Statter) error {
	environment := config.GetEnvironment()
	cfg, err := config.LoadConfigForEnvironment(environment)
	if err != nil {
		log.Fatal(err)
	}
	metricsConfig := metrics.Config{
		Stage:                          cfg.Environment,
		AwsRegion:                      cfg.AWSRegion,
		BufferSize:                     cfg.CloudwatchBufferSize,
		BatchSize:                      cfg.CloudwatchBatchSize,
		FlushInterval:                  time.Second * time.Duration(cfg.CloudwatchFlushIntervalSeconds),
		FlushPollCheckDelay:            time.Millisecond * time.Duration(cfg.CloudwatchFlushCheckDelayMs),
		BufferEmergencyFlushPercentage: cfg.CloudwatchEmergencyFlushPercentage,
	}

	metricLogger, metricsFlusher := metrics.New(metricsConfig)
	go metricsFlusher.FlushMetricsAtInterval()

	clients, err := clients.SetupClients(cfg, metricLogger)
	if err != nil {
		log.Fatalf("Failed to initialize clients: %s", err)
	}

	getTurboCodeAPI := api.GetTurboCodeAPI{
		Statuses: premium.NewPremiumStatuses(*clients, metricLogger),
	}

	getPremiumStatusesAPI := api.GetPremiumStatusesAPI{
		Statuses: premium.NewPremiumStatuses(*clients, metricLogger),
	}

	grantPremiumStatusAPI := api.GrantPremiumStatusAPI{
		IPremiumStatuses: premium.NewPremiumStatuses(*clients, metricLogger),
	}

	cancelPremiumStatusAPI := api.CancelPremiumStatusAPI{
		IPremiumStatuses: premium.NewPremiumStatuses(*clients, metricLogger),
	}

	syncAPI := api.SyncAPI{
		IPremiumStatuses: premium.NewPremiumStatuses(*clients, metricLogger),
	}

	undoSyncAPI := api.UndoSyncAPI{
		IPremiumStatuses: premium.NewPremiumStatuses(*clients, metricLogger),
	}

	isGrandfatheringAPI := api.IsGrandfatheringAPI{
		IGrandfathering: grandfathering.NewGrandfathering(*clients),
	}

	getGrandfatheredUserAPI := api.GetGrandfatheredUserAPI{
		IGrandfathering: grandfathering.NewGrandfathering(*clients),
	}

	createGrandfatheredUserAPI := api.CreateGrandfatheredUserAPI{
		IGrandfathering: grandfathering.NewGrandfathering(*clients),
	}

	updateGrandfatheredUserAPI := api.UpdateGrandfatheredUserAPI{
		IGrandfathering: grandfathering.NewGrandfathering(*clients),
	}

	deleteGrandfatheredUserAPI := api.DeleteGrandfatheredUserAPI{
		IGrandfathering: grandfathering.NewGrandfathering(*clients),
	}

	nitroServer := api.NitroServer{
		GetTurboCodeAPI:            getTurboCodeAPI,
		GetPremiumStatusesAPI:      getPremiumStatusesAPI,
		GrantPremiumStatusAPI:      grantPremiumStatusAPI,
		CancelPremiumStatusAPI:     cancelPremiumStatusAPI,
		IsGrandfatheringAPI:        isGrandfatheringAPI,
		GetGrandfatheredUserAPI:    getGrandfatheredUserAPI,
		CreateGrandfatheredUserAPI: createGrandfatheredUserAPI,
		UpdateGrandfatheredUserAPI: updateGrandfatheredUserAPI,
		DeleteGrandfatheredUserAPI: deleteGrandfatheredUserAPI,
		SyncAPI:                    syncAPI,
		UndoSyncAPI:                undoSyncAPI,
	}

	server, err := nitroServer.NewServer(stats, cfg)
	if err != nil {
		log.Fatal(err)
	}

	log.Info("Starting Nitro server")

	serverConfig := twitchserver.NewConfig()
	serverConfig.Statter = stats
	twitchserver.AddDefaultSignalHandlers()
	return twitchserver.ListenAndServe(server, serverConfig)
}
