package config

import (
	"encoding/json"
	"fmt"
	"os"

	"gopkg.in/validator.v2"
)

const (
	localConfigFilePath  = "/src/code.justin.tv/samus/nitro/config/%s.json"
	globalConfigFilePath = "/etc/nitro/config/%s.json"

	unspecifiedEnvironment = ""
	devEnvironment         = "dev"
)

type Configuration struct {
	IntegrationTestingURI string `validate:"nonzero"`

	AWSRegion   string `validate:"nonzero"`
	Environment string `validate:"nonzero"`

	// Cloudwatch Settings
	CloudwatchBufferSize               int     `validate:"nonzero"`
	CloudwatchBatchSize                int     `validate:"nonzero"`
	CloudwatchFlushIntervalSeconds     int     `validate:"nonzero"`
	CloudwatchFlushCheckDelayMs        int     `validate:"nonzero"`
	CloudwatchEmergencyFlushPercentage float64 `validate:"nonzero"`

	// Client host settings
	BadgesHost   string `validate:"nonzero"`
	MakoHost     string `validate:"nonzero"`
	PaymentsHost string `validate:"nonzero"`

	// Subscriptions Service
	SubscriptionsHost               string `validate:"nonzero"`
	SubscriptionsMaxIdleConnections int    `validate:"nonzero"`

	// Voyager Service
	TwitchVoyagerHost               string `validate:"nonzero"`
	TwitchVoyagerMaxIdleConnections int    `validate:"nonzero"`

	// UserStatus
	GrandfatheredUsersTable  string `validate:"nonzero"`
	GrandfatheringFlagsTable string `validate:"nonzero"`

	// AdditionalPremiumStatusDynamo
	AdditionalPremiumStatusDynamo string `validate:"nonzero"`
}

// GetEnvironment returns environment type
func GetEnvironment() string {
	env := os.Getenv("ENVIRONMENT")
	if env == unspecifiedEnvironment {
		return devEnvironment
	}
	return env
}

// LoadConfigForEnvironment pulls the Config data from the JSON config file
func LoadConfigForEnvironment(environment string) (*Configuration, error) {
	configFilePath, err := getConfigFilePath(environment)
	if err != nil {
		return nil, err
	}
	return loadConfig(configFilePath)
}

func getConfigFilePath(environment string) (string, error) {
	localFname := os.Getenv("GOPATH") + fmt.Sprintf(localConfigFilePath, environment)
	if _, err := os.Stat(localFname); !os.IsNotExist(err) {
		return localFname, nil
	}
	globalFname := fmt.Sprintf(globalConfigFilePath, environment)
	if _, err := os.Stat(globalFname); os.IsNotExist(err) {
		return "", err
	}
	return globalFname, nil
}

func loadConfig(filepath string) (*Configuration, error) {
	file, err := os.Open(filepath)
	if err != nil {
		return nil, fmt.Errorf("Failed to open config file: %v", err)
	}

	decoder := json.NewDecoder(file)
	configuration := &Configuration{}
	if err := decoder.Decode(configuration); err != nil {
		return nil, fmt.Errorf("Failed to parse configuration file: %v", err)
	}

	if err := validator.Validate(configuration); err != nil {
		return nil, fmt.Errorf("Failed to validate configuration file: %v", err)
	}

	return configuration, nil
}
