package integration

import (
	"context"
	"testing"
	"time"

	"fmt"

	"code.justin.tv/samus/nitro/integration_test/config"
	"code.justin.tv/samus/nitro/rpc"
	"github.com/satori/go.uuid"
	. "github.com/smartystreets/goconvey/convey"
)

// Default test users
const qaSamusTurbo = "207069781"
const userIDForGetTest = "test-get"
const userIDForUpdateTest = "test-update"

func TestGrandfatheringAPIs(t *testing.T) {
	client := config.GetNitroClient()

	Convey("Grandfather Clause APIs", t, func() {
		Convey("IsGrandfathering shouldn't fail", func() {
			err := retry(func() error {
				_, err := client.IsGrandfathering(context.TODO(), &nitro.IsGrandfatheringRequest{})
				return err
			})
			So(err, ShouldBeNil)
		})

		Convey("GetGrandfatheredUser shouldn't fail", func() {
			err := retry(func() error {
				_, err := client.GetGrandfatheredUser(context.TODO(), &nitro.GetGrandfatheredUserRequest{
					TwitchUserID: userIDForGetTest,
				})
				return err
			})

			So(err, ShouldBeNil)
		})

		Convey("GetGrandfatheredUser returns the correct data", func() {
			user, err := client.GetGrandfatheredUser(context.TODO(), &nitro.GetGrandfatheredUserRequest{
				TwitchUserID: userIDForGetTest,
			})

			So(err, ShouldBeNil)
			So(user.TwitchUserID, ShouldEqual, userIDForGetTest)
			So(user.HasOldTwitchPrime, ShouldEqual, true)
			So(user.HasNewTwitchPrime, ShouldEqual, true)

			// Hardcoded in the database.
			expectedEndOfOldTwitchPrimeDate := time.Date(3000, 1, 1, 0, 0, 0, 0, time.UTC).
				UTC().
				Format(time.RFC3339)
			So(user.EndOfOldTwitchPrimeDate, ShouldEqual, expectedEndOfOldTwitchPrimeDate)
		})

		Convey("CreateGrandfatheredUser shouldn't fail", func() {
			testID := fmt.Sprintf("test-%s", uuid.NewV4())

			err := retry(func() error {
				_, err := client.CreateGrandfatheredUser(context.TODO(), &nitro.CreateGrandfatheredUserRequest{
					TwitchUserID: testID,
				})
				return err
			})

			So(err, ShouldBeNil)
		})

		Convey("CreateGrandfatheredUser creates an entry with the expected initial fields", func() {
			testID := fmt.Sprintf("test-%s", uuid.NewV4())

			res, err := client.CreateGrandfatheredUser(context.TODO(), &nitro.CreateGrandfatheredUserRequest{
				TwitchUserID: testID,
			})

			So(err, ShouldBeNil)
			So(res.TwitchUserID, ShouldEqual, testID)
			So(res.HasOldTwitchPrime, ShouldBeTrue)
			So(res.HasNewTwitchPrime, ShouldBeTrue)
		})

		Convey("CreateGrandfatheredUser should error on an existing user", func() {
			var err error
			testID := fmt.Sprintf("test-%s", uuid.NewV4())

			_, err = client.CreateGrandfatheredUser(context.TODO(), &nitro.CreateGrandfatheredUserRequest{
				TwitchUserID: testID,
			})

			So(err, ShouldBeNil)

			_, err = client.CreateGrandfatheredUser(context.TODO(), &nitro.CreateGrandfatheredUserRequest{
				TwitchUserID: testID,
			})

			So(err, ShouldNotBeNil)
		})

		Convey("UpdateGrandfatheredUser should work", func() {
			var getRes *nitro.GetGrandfatheredUserResponse
			var err error

			// Initial update

			var hasOldTwitchPrime = true
			var hasNewTwitchPrime = true
			var endOfOldTwitchPrimeDate = time.Date(2018, 9, 14, 5, 0, 0, 0, time.UTC).
				UTC().
				Format(time.RFC3339)

			_, err = client.UpdateGrandfatheredUser(context.TODO(), &nitro.UpdateGrandfatheredUserRequest{
				TwitchUserID:            userIDForUpdateTest,
				HasOldTwitchPrime:       hasOldTwitchPrime,
				HasNewTwitchPrime:       hasNewTwitchPrime,
				EndOfOldTwitchPrimeDate: endOfOldTwitchPrimeDate,
			})

			So(err, ShouldBeNil)

			getRes, err = client.GetGrandfatheredUser(context.TODO(), &nitro.GetGrandfatheredUserRequest{
				TwitchUserID: userIDForUpdateTest,
			})

			So(err, ShouldBeNil)
			So(getRes.HasOldTwitchPrime, ShouldEqual, hasOldTwitchPrime)
			So(getRes.HasNewTwitchPrime, ShouldEqual, hasNewTwitchPrime)
			So(getRes.EndOfOldTwitchPrimeDate, ShouldEqual, endOfOldTwitchPrimeDate)

			// Subsequent update

			hasOldTwitchPrime = false
			hasNewTwitchPrime = false
			endOfOldTwitchPrimeDate = time.Date(2019, 1, 1, 1, 1, 1, 0, time.UTC).
				UTC().
				Format(time.RFC3339)

			_, err = client.UpdateGrandfatheredUser(context.TODO(), &nitro.UpdateGrandfatheredUserRequest{
				TwitchUserID:            userIDForUpdateTest,
				HasOldTwitchPrime:       hasOldTwitchPrime,
				HasNewTwitchPrime:       hasNewTwitchPrime,
				EndOfOldTwitchPrimeDate: endOfOldTwitchPrimeDate,
			})

			So(err, ShouldBeNil)

			getRes, err = client.GetGrandfatheredUser(context.TODO(), &nitro.GetGrandfatheredUserRequest{
				TwitchUserID: userIDForUpdateTest,
			})

			So(getRes.HasOldTwitchPrime, ShouldEqual, hasOldTwitchPrime)
			So(getRes.HasNewTwitchPrime, ShouldEqual, hasNewTwitchPrime)
			So(getRes.EndOfOldTwitchPrimeDate, ShouldEqual, endOfOldTwitchPrimeDate)
		})

		Convey("DeleteGrandfatheredUser should work", func() {
			var err error
			testID := fmt.Sprintf("test-%s", uuid.NewV4())

			_, err = client.CreateGrandfatheredUser(context.TODO(), &nitro.CreateGrandfatheredUserRequest{
				TwitchUserID: testID,
			})

			So(err, ShouldBeNil)

			_, err = client.DeleteGrandfatheredUser(context.TODO(), &nitro.DeleteGrandfatheredUserRequest{
				TwitchUserID: testID,
			})

			So(err, ShouldBeNil)
		})

	})
}

const defaultAttempts = 3
const backoff = 2

func retry(fn func() error) error {
	return _retry(defaultAttempts, time.Second, fn)
}

func _retry(attempts int, timeout time.Duration, fn func() error) error {
	err := fn()
	if err != nil {
		if attempts--; attempts > 0 {
			time.Sleep(timeout)
			return _retry(attempts, timeout*backoff, fn)
		}
	}
	return err
}
