package integration_test

import (
	"context"
	"testing"

	voyagertwirp "code.justin.tv/amzn/TwitchVoyagerTwirp"
	"code.justin.tv/samus/nitro/integration_test/config"
	nitro "code.justin.tv/samus/nitro/rpc"
	"github.com/golang/protobuf/ptypes"
	. "github.com/smartystreets/goconvey/convey"
)

const (
	// Default test users
	ivoneProvolone   = "207375363" // Test user without Prime
	diegoManchego    = "207412330" // The OG test user & not enrolled into Presto
	tiagoAsiago      = "216569891" // Test user for sync / undo sync
	grettaFeta       = "250256873" // Test user with only new Prime & Enrolled into Presto
	simoneMascarpone = "253204424" // Grandfathered test user
	qaSamusTurbo     = "207069781" // Enrolled into Presto
	qa_no_prime      = "229523641"

	// Twitch Prime product IDs
	oldTwitchPrimePID = "12658"
	newTwitchPrimePID = "996688"

	// Default end date for old Twitch Prime product
	defaultTime = "2030-01-02T15:04:05Z"
)

func TestPremiumStatuses(t *testing.T) {
	nitroClient := config.GetNitroClient()
	voyagerClient := config.GetVoyagerClient()

	Convey("Testing GetPremiumStatuses API", t, func() {

		SkipConvey("for a Turbo user", func() {
			req := &nitro.GetPremiumStatusesRequest{
				TwitchUserID: qaSamusTurbo,
			}

			resp, err := nitroClient.GetPremiumStatuses(context.TODO(), req)

			So(err, ShouldBeNil)
			So(resp.HasPrime, ShouldBeFalse)
			So(resp.HasTurbo, ShouldBeTrue)
			So(resp.HasPresto, ShouldBeTrue)
		})

		Convey("for a old Twitch Prime user", func() {
			req := &nitro.GetPremiumStatusesRequest{
				TwitchUserID: diegoManchego,
			}

			resp, err := nitroClient.GetPremiumStatuses(context.TODO(), req)

			So(err, ShouldBeNil)
			So(resp.HasPrime, ShouldBeTrue)
			So(resp.HasTurbo, ShouldBeFalse)
			So(resp.HasPresto, ShouldBeFalse)
		})

		Convey("for a new Twitch Prime user", func() {
			req := &nitro.GetPremiumStatusesRequest{
				TwitchUserID: grettaFeta,
			}

			resp, err := nitroClient.GetPremiumStatuses(context.TODO(), req)

			So(err, ShouldBeNil)
			So(resp.HasPrime, ShouldBeTrue)
			So(resp.HasTurbo, ShouldBeFalse)
			So(resp.HasPresto, ShouldBeTrue)
		})

		Convey("with a non-Twitch Prime, non-Turbo user", func() {
			req := &nitro.GetPremiumStatusesRequest{
				TwitchUserID: qa_no_prime,
			}

			resp, err := nitroClient.GetPremiumStatuses(context.TODO(), req)

			So(err, ShouldBeNil)
			So(resp.HasPrime, ShouldBeFalse)
			So(resp.HasTurbo, ShouldBeFalse)
			So(resp.HasPresto, ShouldBeFalse)
		})
	})

	Convey("Testing GrantPremiumStatus API", t, func() {
		Convey("granting Twitch Prime", func() {
			req := &nitro.GrantPremiumStatusRequest{
				TwitchUserID: ivoneProvolone,
				ProductName:  nitro.PremiumProduct_PRIME,
			}

			resp, err := nitroClient.GrantPremiumStatus(context.TODO(), req)
			So(err, ShouldBeNil)
			So(resp.HasPrime, ShouldBeTrue)
		})

		Convey("granting Twitch Prime for the second time", func() {
			req := &nitro.GrantPremiumStatusRequest{
				TwitchUserID: ivoneProvolone,
				ProductName:  nitro.PremiumProduct_PRIME,
			}

			resp, err := nitroClient.GrantPremiumStatus(context.TODO(), req)
			So(err, ShouldBeNil)
			So(resp.HasPrime, ShouldBeTrue)
		})

		Convey("granting Presto Prime", func() {
			req := &nitro.GrantPremiumStatusRequest{
				TwitchUserID: ivoneProvolone,
				ProductName:  nitro.PremiumProduct_PRESTO,
			}

			resp, err := nitroClient.GrantPremiumStatus(context.TODO(), req)
			So(err, ShouldBeNil)
			So(resp.HasPresto, ShouldBeTrue)
		})

		Convey("granting Presto Prime for the second time", func() {
			req := &nitro.GrantPremiumStatusRequest{
				TwitchUserID: ivoneProvolone,
				ProductName:  nitro.PremiumProduct_PRESTO,
			}

			resp, err := nitroClient.GrantPremiumStatus(context.TODO(), req)
			So(err, ShouldBeNil)
			So(resp.HasPresto, ShouldBeTrue)
		})
	})

	// NOTE: Make sure to cancel the status for the same test user that you chose to grant
	Convey("Testing CancelPremiumStatus API", t, func() {
		Convey("cancelling Twitch Prime", func() {
			req := &nitro.CancelPremiumStatusRequest{
				TwitchUserID: ivoneProvolone,
				ProductName:  nitro.PremiumProduct_PRIME,
			}

			resp, err := nitroClient.CancelPremiumStatus(context.TODO(), req)
			So(err, ShouldBeNil)
			So(resp.HasPrime, ShouldBeFalse)
		})

		Convey("cancelling Presto Prime", func() {
			req := &nitro.CancelPremiumStatusRequest{
				TwitchUserID: ivoneProvolone,
				ProductName:  nitro.PremiumProduct_PRESTO,
			}

			resp, err := nitroClient.CancelPremiumStatus(context.TODO(), req)
			So(err, ShouldBeNil)
			So(resp.HasPresto, ShouldBeFalse)
		})

		Convey("cancelling Presto Prime for the second time", func() {
			req := &nitro.CancelPremiumStatusRequest{
				TwitchUserID: ivoneProvolone,
				ProductName:  nitro.PremiumProduct_PRESTO,
			}

			resp, err := nitroClient.CancelPremiumStatus(context.TODO(), req)
			So(err, ShouldBeNil)
			So(resp.HasPresto, ShouldBeFalse)
		})
	})

	Convey("Testing Sync / UndoSync API", t, func() {
		SkipConvey("syncing a member with only old prime", func() {
			syncReq := &nitro.SyncRequest{
				TwitchUserID:            tiagoAsiago,
				EndOfOldTwitchPrimeDate: defaultTime,
			}

			_, syncErr := nitroClient.Sync(context.TODO(), syncReq)
			So(syncErr, ShouldBeNil)

			// Ensure products exist for new and old Twitch Prime
			getUserProductSubsReq := &voyagertwirp.GetUserProductSubscriptionsRequest{
				UserId:     tiagoAsiago,
				ProductIds: []string{oldTwitchPrimePID, newTwitchPrimePID},
			}

			getUserProductSubsResp, getUserSubsErr := voyagerClient.GetUserProductSubscriptions(context.TODO(), getUserProductSubsReq)
			So(getUserSubsErr, ShouldBeNil)
			So(getUserProductSubsResp.GetSubscriptions(), ShouldNotBeNil)
			So(len(getUserProductSubsResp.GetSubscriptions()), ShouldEqual, 2)

			// Ensure old Twitch Prime product has an end date set to it
			var oldTwitchPrimeSub voyagertwirp.Subscription
			for _, subscription := range getUserProductSubsResp.GetSubscriptions() {
				productId := subscription.ProductId
				if productId == oldTwitchPrimePID {
					oldTwitchPrimeSub = *subscription
				}
			}

			strBenefitEnd := ptypes.TimestampString(oldTwitchPrimeSub.End)
			So(strBenefitEnd, ShouldNotBeBlank)
			So(strBenefitEnd, ShouldEqual, defaultTime)

			// Undo sync for repeated testing
			undoReq := &nitro.UndoSyncRequest{
				TwitchUserID: tiagoAsiago,
			}

			_, undoErr := nitroClient.UndoSync(context.TODO(), undoReq)
			So(undoErr, ShouldBeNil)
		})
	})
}
