package premium

import (
	"context"
	"strings"
	"time"

	payments "code.justin.tv/revenue/payments-service-go-client/client"
	"code.justin.tv/samus/nitro/internal/clients"
	"code.justin.tv/samus/nitro/metrics"
	nitro "code.justin.tv/samus/nitro/rpc"
	"github.com/pkg/errors"
	log "github.com/sirupsen/logrus"
)

type PremiumStatuses interface {
	GetPremiumStatuses(twitchUserID string) (*Statuses, error)
	GrantPremiumStatus(ctx context.Context, userID string, product nitro.PremiumProduct) (*Statuses, error)
	CancelPremiumStatus(ctx context.Context, userID string, product nitro.PremiumProduct) (*Statuses, error)
	GetTurboCode(twitchUserID string) (string, error)

	Sync(ctx context.Context, userID string, time time.Time) error
	UndoSync(ctx context.Context, userID string) error
}

type PremiumStatusesImpl struct {
	Clients      clients.Clients
	MetricLogger metrics.IMetricLogger
}

type Statuses struct {
	HasTurbo  bool
	HasPrime  bool
	HasPresto bool
}

const (
	// Product IDs
	turbo        = "324"
	turboYearly  = "388"
	twitchPrime  = "12658" // Ad-free Prime
	twitchPrime2 = "996688"
)

// Product IDs with Premium Status
var PRODUCT_IDS = []string{turbo, turboYearly, twitchPrime, twitchPrime2}

var TURBO_PRODUCT_IDS = map[string]bool{
	turbo:       true,
	turboYearly: true,
}

var PRIME_PRODUCT_IDS = map[string]bool{
	twitchPrime:  true, // Ad-free Prime
	twitchPrime2: true,
}

var PRODUCT_SHORT_NAMES_BY_ID = map[string]string{
	twitchPrime:  "twitch_prime",
	twitchPrime2: "twitchprime",
	turbo:        "turbo",
	turboYearly:  "turbo_yearly",
}

const samusPaymentProvider = "samus"

func NewPremiumStatuses(c clients.Clients, m metrics.IMetricLogger) PremiumStatuses {
	return &PremiumStatusesImpl{
		Clients:      c,
		MetricLogger: m,
	}
}

// GetPremiumStatuses contains the business logic to determine if a user has Prime, Turbo & Presto
func (s *PremiumStatusesImpl) GetPremiumStatuses(twitchUserID string) (*Statuses, error) {

	logger := log.WithFields(log.Fields{
		"userID": twitchUserID,
	})

	subsResp, err := s.Clients.SubsServiceClient.GetUserProductSubscriptions(twitchUserID, PRODUCT_IDS)
	if err != nil {
		return nil, errors.Wrap(err, "Unable to get premium statuses: Failed to get user product subscriptions")
	}

	returnStatuses := Statuses{
		HasTurbo:  false,
		HasPrime:  false,
		HasPresto: false,
	}

	additionalPremiumStatusDynamo, err := s.Clients.AdditionalPremiumStatusDynamoDao.Get(twitchUserID)

	if err != nil {
		s.MetricLogger.LogCountMetric("GetPremiumStatusesForPrestoFailure", 1)
		logger.Error("Error while retrieving the Presto status for the customer: ", err)
	}

	if additionalPremiumStatusDynamo != nil && strings.EqualFold(additionalPremiumStatusDynamo.PrestoEnable, "true") {
		returnStatuses.HasPresto = true
	}

	for _, subscription := range subsResp.GetSubscriptions() {
		productId := subscription.ProductId

		if TURBO_PRODUCT_IDS[productId] {
			returnStatuses.HasTurbo = true
		}

		if PRIME_PRODUCT_IDS[productId] {
			returnStatuses.HasPrime = true
		}
	}

	return &returnStatuses, nil
}

func (s *PremiumStatusesImpl) GrantPremiumStatus(ctx context.Context, userID string, product nitro.PremiumProduct) (*Statuses, error) {
	// NOTE: GrantPremiumStatus only grants Prime & Presto, not Turbo
	if product != nitro.PremiumProduct_PRIME && product != nitro.PremiumProduct_PRESTO {
		return nil, errors.New("Unable to grant premium status: No support for given product")
	}

	if product == nitro.PremiumProduct_PRESTO {

		err := s.Clients.AdditionalPremiumStatusDynamoDao.Put(userID)
		if err != nil {
			return nil, errors.Wrap(err, "Unable to grant premium status: for user "+userID+" product Presto.")
		}
		return &Statuses{HasPresto: true}, nil
	}

	// Check if user already has Prime
	statuses, getErr := s.GetPremiumStatuses(userID)
	if getErr != nil {
		return nil, errors.Wrap(getErr, "Unable to grant premium status: Failed to get premium status")
	}

	if !statuses.HasPrime {
		// Always grant new Prime
		_, profileErr := s.Clients.PaymentsClient.CompletePurchase(ctx, userID, PRODUCT_SHORT_NAMES_BY_ID[twitchPrime2], payments.CompletePurchaseRequest{PaymentProvider: samusPaymentProvider})
		if profileErr != nil {
			return nil, errors.Wrap(profileErr, "Unable to grant premium status: Failed to complete purchase for user "+userID+" product "+twitchPrime2)
		}
	}

	// Grant prime badge
	badgesErr := s.Clients.BadgesClient.GrantPrimeBadge(ctx, userID)
	if badgesErr != nil {
		return nil, errors.Wrap(badgesErr, "Unable to grant premium status: Failed to grant prime badge")
	}

	return &Statuses{HasPrime: true}, nil
}

func (s *PremiumStatusesImpl) CancelPremiumStatus(ctx context.Context, userID string, product nitro.PremiumProduct) (*Statuses, error) {
	// NOTE: CancelPremiumStatus only cancels Prime & Presto, not Turbo
	if product != nitro.PremiumProduct_PRIME && product != nitro.PremiumProduct_PRESTO {
		return nil, errors.New("Unable to cancel premium status: No support for given product")
	}

	if product == nitro.PremiumProduct_PRESTO {

		err := s.Clients.AdditionalPremiumStatusDynamoDao.Delete(userID)
		if err != nil {
			return nil, errors.Wrap(err, "Unable to cancel premium status: for user "+userID+" product Presto.")
		}
		return &Statuses{HasPresto: false}, nil
	}

	// Check if user already doesn't have Prime
	statuses, getErr := s.GetPremiumStatuses(userID)
	if getErr != nil {
		return nil, errors.Wrap(getErr, "Unable to cancel premium status: Failed to get premium status")
	}

	if !statuses.HasPrime {
		return &Statuses{HasPrime: false}, nil
	}

	// Cancel ticket product
	subsResp, subsErr := s.Clients.SubsServiceClient.GetUserProductSubscriptions(userID, PRODUCT_IDS)
	if subsErr != nil {
		return nil, errors.Wrap(subsErr, "Unable to cancel premium status: Failed to get user product subscriptions")
	}

	for _, subscription := range subsResp.GetSubscriptions() {
		productId := subscription.ProductId

		// If user has any Twitch Prime, cancel the purchase (which cancels the ticket)
		if PRIME_PRODUCT_IDS[productId] {
			purchaseErr := s.Clients.PaymentsClient.CancelPurchase(ctx, userID, PRODUCT_SHORT_NAMES_BY_ID[productId], payments.CancelPurchaseRequest{PurchaseProfileID: subscription.OriginId})

			if purchaseErr != nil {
				return nil, errors.Wrap(purchaseErr, "Unable to cancel premium status: Failed to cancel purchase")
			}
		}
	}

	// Remove Prime badge
	badgesErr := s.Clients.BadgesClient.RemovePrimeBadge(ctx, userID)
	if badgesErr != nil {
		return nil, errors.Wrap(badgesErr, "Unable to cancel premium status: Failed to remove prime badge")
	}

	// Reset smilies
	_, smiliesErr := s.Clients.MakoClient.ResetSmiliesSetForUser(ctx, userID)
	if smiliesErr != nil {
		return nil, errors.Wrap(smiliesErr, "Unable to cancel premium status: Failed to reset smilies")
	}

	return &Statuses{HasPrime: false}, nil
}

func (s *PremiumStatusesImpl) GetTurboCode(twitchUserID string) (string, error) {
	subsResponse, err := s.Clients.SubsServiceClient.GetUserProductSubscriptions(twitchUserID, PRODUCT_IDS)
	if err != nil {
		return "", errors.Wrap(err, "Unable to get turbo code: Failed to get user product subscriptions")
	}

	subscriptions := subsResponse.GetSubscriptions()
	for _, subscription := range subscriptions {
		if TURBO_PRODUCT_IDS[subscription.ProductId] {
			return subscription.GetAccessGuid(), nil
		}
	}

	return "", nil
}

func (s *PremiumStatusesImpl) Sync(ctx context.Context, userID string, endOfOldTwitchPrimeDate time.Time) error {
	// Gather a user's set of Prime related subscriptions
	subsResp, err := s.Clients.SubsServiceClient.GetUserProductSubscriptions(userID, PRODUCT_IDS)
	if err != nil {
		return errors.Wrap(err, "Unable to sync: Failed to get user product subscriptions")
	}

	hasOldPrime := false
	hasNewPrime := false

	var userOldPrimeSubIDs []string

	for _, subscription := range subsResp.GetSubscriptions() {
		productId := subscription.ProductId

		if productId == twitchPrime {
			hasOldPrime = true
			userOldPrimeSubIDs = append(userOldPrimeSubIDs, subscription.Id)
		}

		if productId == twitchPrime2 {
			hasNewPrime = true
		}
	}

	if hasOldPrime {
		// Case 1: Has old Prime, missing new Prime. This backfills existing users with the new Prime.
		if !hasNewPrime {
			// Grant new Prime in addition to ad-free Prime
			_, profileErr := s.Clients.PaymentsClient.CompletePurchase(ctx, userID, PRODUCT_SHORT_NAMES_BY_ID[twitchPrime2], payments.CompletePurchaseRequest{PaymentProvider: samusPaymentProvider})
			if profileErr != nil {
				return errors.Wrap(profileErr, "Unable to sync: Failed to complete purchase for user "+userID+" product "+twitchPrime)
			}
		}

		// Update end date for ad-free Prime subscription
		for _, oldPrimeSubID := range userOldPrimeSubIDs {
			updateTicketErr := s.Clients.SubsServiceClient.UpdateTicketEndDate(oldPrimeSubID, endOfOldTwitchPrimeDate)
			if updateTicketErr != nil {
				return errors.Wrap(updateTicketErr, "Unable to sync: Failed to complete update ticket end date for user "+userID+" product id "+twitchPrime)
			}
		}
	}

	return nil
}

func (s *PremiumStatusesImpl) UndoSync(ctx context.Context, userID string) error {
	// Cancel new Prime subscription
	subsResp, subsErr := s.Clients.SubsServiceClient.GetUserProductSubscriptions(userID, PRODUCT_IDS)
	if subsErr != nil {
		return errors.Wrap(subsErr, "Unable to undo sync: Failed to get user product subscriptions")
	}

	for _, subscription := range subsResp.GetSubscriptions() {
		productId := subscription.ProductId

		if productId == twitchPrime2 {
			purchaseErr := s.Clients.PaymentsClient.CancelPurchase(ctx, userID, PRODUCT_SHORT_NAMES_BY_ID[productId], payments.CancelPurchaseRequest{PurchaseProfileID: subscription.OriginId})

			if purchaseErr != nil {
				return errors.Wrap(purchaseErr, "Unable to cancel premium status: Failed to cancel purchase")
			}
		}
	}

	return nil
}
