package premium

import (
	"context"
	"testing"

	"code.justin.tv/samus/nitro/internal/clients/dynamo"

	voyagertwirp "code.justin.tv/amzn/TwitchVoyagerTwirp"
	mako_client "code.justin.tv/commerce/mako-client"
	payments "code.justin.tv/revenue/payments-service-go-client/client"
	"code.justin.tv/samus/nitro/_tools/src/github.com/pkg/errors"
	mocks "code.justin.tv/samus/nitro/clients"
	"code.justin.tv/samus/nitro/internal/clients"
	. "code.justin.tv/samus/nitro/internal/clients/badges"
	. "code.justin.tv/samus/nitro/internal/clients/dynamo/mocks"
	. "code.justin.tv/samus/nitro/internal/clients/payments"
	. "code.justin.tv/samus/nitro/internal/clients/subs"
	. "code.justin.tv/samus/nitro/internal/clients/user_status"
	"code.justin.tv/samus/nitro/metrics"

	nitro "code.justin.tv/samus/nitro/rpc"
	. "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"
)

func TestPremiumStatuses(t *testing.T) {
	Convey("Test PremiumStatuses APIs", t, func() {
		mockIBadgesClient := new(MockIBadgesClient)
		mockIPaymentsClient := new(MockIPaymentsClient)
		mockIMakoClient := new(mocks.IMakoClient)
		mockISubServiceClient := new(MockISubServiceClient)
		mockIUserStatusClient := new(MockIUserStatusClient)
		mockAdditionalPremiumStatusDynamoDao := new(IAdditionalPremiumStatusDynamoDao)
		mockMetricLogger := new(metrics.MockIMetricLogger)

		mockClients := clients.Clients{
			BadgesClient:                     mockIBadgesClient,
			MakoClient:                       mockIMakoClient,
			PaymentsClient:                   mockIPaymentsClient,
			SubsServiceClient:                mockISubServiceClient,
			UserStatusClient:                 mockIUserStatusClient,
			AdditionalPremiumStatusDynamoDao: mockAdditionalPremiumStatusDynamoDao,
		}

		premiumStatuses := PremiumStatusesImpl{
			Clients:      mockClients,
			MetricLogger: mockMetricLogger,
		}

		mockTurboMonthlySubscription := voyagertwirp.Subscription{
			ProductId: turbo,
		}

		mockTurboYearlySubscription := voyagertwirp.Subscription{
			ProductId: turboYearly,
		}

		mockPrimeSubscription := voyagertwirp.Subscription{
			ProductId: twitchPrime,
		}

		mockPrime2Subscription := voyagertwirp.Subscription{
			ProductId: twitchPrime2,
		}

		userID := "fake-user-id"

		Convey("Test GetPremiumStatuses API", func() {
			Convey("returns hasPrime = true, hasTurbo = false and hasPresto = true if a user has only the ad-free (original) prime subscription and presto enrolled", func() {
				res := voyagertwirp.GetUserProductSubscriptionsResponse{
					Subscriptions: []*voyagertwirp.Subscription{&mockPrimeSubscription},
				}

				mockAdditionalPremiumStatusDynamo := dynamo.AdditionalPremiumStatusDynamo{
					TwitchUserID: userID,
					PrestoEnable: "true",
				}

				mockISubServiceClient.On("GetUserProductSubscriptions", userID, mock.Anything).Return(&res, nil)
				mockAdditionalPremiumStatusDynamoDao.On("Get", userID).Return(&mockAdditionalPremiumStatusDynamo, nil)

				result, err := premiumStatuses.GetPremiumStatuses(userID)
				So(err, ShouldBeNil)
				So(result.HasPrime, ShouldEqual, true)
				So(result.HasTurbo, ShouldEqual, false) // Always have turbo if you have ad-free prime
				So(result.HasPresto, ShouldEqual, true)
			})

			Convey("returns hasPrime = true, hasTurbo = false and hasPresto = false if a user has only the new prime subscription and not enrolled in presto", func() {
				res := voyagertwirp.GetUserProductSubscriptionsResponse{
					Subscriptions: []*voyagertwirp.Subscription{&mockPrime2Subscription},
				}

				mockAdditionalPremiumStatusDynamo := dynamo.AdditionalPremiumStatusDynamo{
					TwitchUserID: userID,
					PrestoEnable: "false",
				}

				mockISubServiceClient.On("GetUserProductSubscriptions", userID, mock.Anything).Return(&res, nil)
				mockAdditionalPremiumStatusDynamoDao.On("Get", userID).Return(&mockAdditionalPremiumStatusDynamo, nil)

				result, err := premiumStatuses.GetPremiumStatuses(userID)
				So(err, ShouldBeNil)
				So(result.HasPrime, ShouldEqual, true)
				So(result.HasTurbo, ShouldEqual, false) // Prime 2 removes the ad-free benefit
				So(result.HasPresto, ShouldEqual, false)
			})

			Convey("returns hasTurbo = true, hasPrime = false and hasPresto = true if a user has only the turbo monthly subscription and presto enrolled", func() {
				res := voyagertwirp.GetUserProductSubscriptionsResponse{
					Subscriptions: []*voyagertwirp.Subscription{&mockTurboMonthlySubscription},
				}

				mockAdditionalPremiumStatusDynamo := dynamo.AdditionalPremiumStatusDynamo{
					TwitchUserID: userID,
					PrestoEnable: "true",
				}

				mockISubServiceClient.On("GetUserProductSubscriptions", userID, mock.Anything).Return(&res, nil)
				mockAdditionalPremiumStatusDynamoDao.On("Get", userID).Return(&mockAdditionalPremiumStatusDynamo, nil)

				result, err := premiumStatuses.GetPremiumStatuses(userID)
				So(err, ShouldBeNil)
				So(result.HasTurbo, ShouldEqual, true)
				So(result.HasPrime, ShouldEqual, false)
				So(result.HasPresto, ShouldEqual, true)
			})

			Convey("returns hasTurbo = true, hasPrime = false and and hasPresto = false if a user has only the turbo yearly subscription and not enrolled in presto", func() {
				res := voyagertwirp.GetUserProductSubscriptionsResponse{
					Subscriptions: []*voyagertwirp.Subscription{&mockTurboYearlySubscription},
				}

				mockAdditionalPremiumStatusDynamo := dynamo.AdditionalPremiumStatusDynamo{
					TwitchUserID: userID,
					PrestoEnable: "false",
				}

				mockISubServiceClient.On("GetUserProductSubscriptions", userID, mock.Anything).Return(&res, nil)
				mockAdditionalPremiumStatusDynamoDao.On("Get", userID).Return(&mockAdditionalPremiumStatusDynamo, nil)

				result, err := premiumStatuses.GetPremiumStatuses(userID)
				So(err, ShouldBeNil)
				So(result.HasPrime, ShouldEqual, false)
				So(result.HasTurbo, ShouldEqual, true)
				So(result.HasPresto, ShouldEqual, false)
			})

			Convey("returns hasTurbo = false, hasPrime = false and hasPresto = false if a user has no subscriptions and not enrolled in presto", func() {
				res := voyagertwirp.GetUserProductSubscriptionsResponse{
					Subscriptions: []*voyagertwirp.Subscription{},
				}

				mockAdditionalPremiumStatusDynamo := dynamo.AdditionalPremiumStatusDynamo{
					TwitchUserID: userID,
					PrestoEnable: "false",
				}

				mockISubServiceClient.On("GetUserProductSubscriptions", userID, mock.Anything).Return(&res, nil)
				mockAdditionalPremiumStatusDynamoDao.On("Get", userID).Return(&mockAdditionalPremiumStatusDynamo, nil)

				result, err := premiumStatuses.GetPremiumStatuses(userID)
				So(err, ShouldBeNil)
				So(result.HasPrime, ShouldEqual, false)
				So(result.HasTurbo, ShouldEqual, false)
				So(result.HasPresto, ShouldEqual, false)
			})

			Convey("returns error when subs service has an error", func() {
				mockISubServiceClient.On("GetUserProductSubscriptions", mock.Anything, mock.Anything).Return(nil, errors.New("Error calling subs service"))

				_, err := premiumStatuses.GetPremiumStatuses("id")
				So(err, ShouldNotBeNil)
			})

			Convey("returns presto status as false when AdditionalPremiumStatusDynamoDao service has an error", func() {

				res := voyagertwirp.GetUserProductSubscriptionsResponse{
					Subscriptions: []*voyagertwirp.Subscription{},
				}
				mockISubServiceClient.On("GetUserProductSubscriptions", "id", mock.Anything).Return(&res, nil)
				mockAdditionalPremiumStatusDynamoDao.On("Get", mock.Anything).Return(nil, errors.New("Error calling subs service"))
				mockMetricLogger.On("LogCountMetric", "GetPremiumStatusesForPrestoFailure", 1.0).Return()

				result, err := premiumStatuses.GetPremiumStatuses("id")
				So(err, ShouldBeNil)
				So(result.HasPrime, ShouldEqual, false)
				So(result.HasTurbo, ShouldEqual, false)
				So(result.HasPresto, ShouldEqual, false)
			})
		})

		Convey("Test GrantPremiumStatuses API", func() {
			Convey("granting Prime when user already has Prime", func() {
				subsResp := voyagertwirp.GetUserProductSubscriptionsResponse{
					Subscriptions: []*voyagertwirp.Subscription{&mockPrimeSubscription},
				}

				mockAdditionalPremiumStatusDynamo := dynamo.AdditionalPremiumStatusDynamo{
					TwitchUserID: userID,
					PrestoEnable: "true",
				}

				mockAdditionalPremiumStatusDynamoDao.On("Get", userID).Return(&mockAdditionalPremiumStatusDynamo, nil)
				mockISubServiceClient.On("GetUserProductSubscriptions", userID, PRODUCT_IDS).Return(&subsResp, nil)
				mockIUserStatusClient.On("GetGrandfatheredUser", userID).Return(nil, nil)
				mockIBadgesClient.On("GrantPrimeBadge", context.TODO(), userID).Return(nil)

				resp, err := premiumStatuses.GrantPremiumStatus(context.TODO(), userID, nitro.PremiumProduct_PRIME)
				So(err, ShouldBeNil)
				So(resp.HasPrime, ShouldBeTrue)
			})

			Convey("granting new Prime with success", func() {
				subsResp := voyagertwirp.GetUserProductSubscriptionsResponse{
					Subscriptions: []*voyagertwirp.Subscription{},
				}

				paymentsReq := payments.CompletePurchaseRequest{
					PaymentProvider: samusPaymentProvider,
				}

				mockAdditionalPremiumStatusDynamo := dynamo.AdditionalPremiumStatusDynamo{
					TwitchUserID: userID,
					PrestoEnable: "true",
				}

				mockAdditionalPremiumStatusDynamoDao.On("Get", userID).Return(&mockAdditionalPremiumStatusDynamo, nil)
				mockISubServiceClient.On("GetUserProductSubscriptions", userID, PRODUCT_IDS).Return(&subsResp, nil)
				mockIPaymentsClient.On("CompletePurchase", context.TODO(), userID, PRODUCT_SHORT_NAMES_BY_ID[twitchPrime2], paymentsReq).Return(&payments.PurchaseProfile{}, nil)
				mockIBadgesClient.On("GrantPrimeBadge", context.TODO(), userID).Return(nil)

				resp, err := premiumStatuses.GrantPremiumStatus(context.TODO(), userID, nitro.PremiumProduct_PRIME)
				So(err, ShouldBeNil)
				So(resp.HasPrime, ShouldBeTrue)
			})

			Convey("granting Prime with failure", func() {

				mockAdditionalPremiumStatusDynamo := dynamo.AdditionalPremiumStatusDynamo{
					TwitchUserID: userID,
					PrestoEnable: "true",
				}

				mockAdditionalPremiumStatusDynamoDao.On("Get", userID).Return(&mockAdditionalPremiumStatusDynamo, nil)
				mockISubServiceClient.On("GetUserProductSubscriptions", userID, PRODUCT_IDS).Return(nil, errors.New("Failed to get user products"))

				resp, err := premiumStatuses.GrantPremiumStatus(context.TODO(), userID, nitro.PremiumProduct_PRIME)
				So(err, ShouldNotBeNil)
				So(resp, ShouldBeNil)
			})

			Convey("Granting Presto to the user", func() {
				mockAdditionalPremiumStatusDynamoDao.On("Put", userID).Return(nil)

				resp, err := premiumStatuses.GrantPremiumStatus(context.TODO(), userID, nitro.PremiumProduct_PRESTO)
				So(err, ShouldBeNil)
				So(resp.HasPresto, ShouldBeTrue)
			})

			Convey("Failure while granting Presto to the user", func() {
				mockAdditionalPremiumStatusDynamoDao.On("Put", userID).Return(errors.New("Failed to grant the Presto to the user"))

				resp, err := premiumStatuses.GrantPremiumStatus(context.TODO(), userID, nitro.PremiumProduct_PRESTO)
				So(err, ShouldNotBeNil)
				So(resp, ShouldBeNil)
			})
		})

		Convey("Test CancelPremiumStatuses API", func() {
			Convey("cancelling Prime when user already doesn't have Prime", func() {
				subsResp := voyagertwirp.GetUserProductSubscriptionsResponse{
					Subscriptions: []*voyagertwirp.Subscription{},
				}

				mockAdditionalPremiumStatusDynamo := dynamo.AdditionalPremiumStatusDynamo{
					TwitchUserID: userID,
					PrestoEnable: "true",
				}

				mockAdditionalPremiumStatusDynamoDao.On("Get", userID).Return(&mockAdditionalPremiumStatusDynamo, nil)
				mockISubServiceClient.On("GetUserProductSubscriptions", userID, PRODUCT_IDS).Return(&subsResp, nil)

				resp, err := premiumStatuses.CancelPremiumStatus(context.TODO(), userID, nitro.PremiumProduct_PRIME)
				So(err, ShouldBeNil)
				So(resp.HasPrime, ShouldBeFalse)
			})

			Convey("cancelling Prime with success", func() {
				subsResp := voyagertwirp.GetUserProductSubscriptionsResponse{
					Subscriptions: []*voyagertwirp.Subscription{&mockPrimeSubscription, &mockPrime2Subscription},
				}

				// Request for cancelling ad-free Prime
				paymentsPrimeReq := payments.CancelPurchaseRequest{
					PurchaseProfileID: mockPrimeSubscription.OriginId,
				}
				// Request for cancelling new Prime
				paymentsPrime2Req := payments.CancelPurchaseRequest{
					PurchaseProfileID: mockPrime2Subscription.OriginId,
				}

				mockAdditionalPremiumStatusDynamo := dynamo.AdditionalPremiumStatusDynamo{
					TwitchUserID: userID,
					PrestoEnable: "true",
				}

				mockAdditionalPremiumStatusDynamoDao.On("Get", userID).Return(&mockAdditionalPremiumStatusDynamo, nil)
				mockISubServiceClient.On("GetUserProductSubscriptions", userID, PRODUCT_IDS).Return(&subsResp, nil)
				mockIPaymentsClient.On("CancelPurchase", context.TODO(), userID, PRODUCT_SHORT_NAMES_BY_ID[twitchPrime], paymentsPrimeReq).Return(nil)
				mockIPaymentsClient.On("CancelPurchase", context.TODO(), userID, PRODUCT_SHORT_NAMES_BY_ID[twitchPrime2], paymentsPrime2Req).Return(nil)
				mockIBadgesClient.On("RemovePrimeBadge", context.TODO(), userID).Return(nil)
				mockIMakoClient.On("ResetSmiliesSetForUser", context.TODO(), userID).Return(&mako_client.SetSmiliesResponse{}, nil)

				resp, err := premiumStatuses.CancelPremiumStatus(context.TODO(), userID, nitro.PremiumProduct_PRIME)
				So(err, ShouldBeNil)
				So(resp.HasPrime, ShouldBeFalse)
			})

			Convey("cancelling Prime with failure", func() {
				mockISubServiceClient.On("GetUserProductSubscriptions", userID, PRODUCT_IDS).Return(nil, errors.New("Failed to get user products"))

				resp, err := premiumStatuses.CancelPremiumStatus(context.TODO(), userID, nitro.PremiumProduct_PRIME)
				So(err, ShouldNotBeNil)
				So(resp, ShouldBeNil)
			})

			Convey("Cancelling Presto to the user", func() {
				mockAdditionalPremiumStatusDynamoDao.On("Delete", userID).Return(nil)

				resp, err := premiumStatuses.CancelPremiumStatus(context.TODO(), userID, nitro.PremiumProduct_PRESTO)
				So(err, ShouldBeNil)
				So(resp.HasPresto, ShouldBeFalse)
			})

			Convey("Failure while cancelling Presto to the user", func() {
				mockAdditionalPremiumStatusDynamoDao.On("Delete", userID).Return(errors.New("Failed to cancel the Presto to the user"))

				resp, err := premiumStatuses.CancelPremiumStatus(context.TODO(), userID, nitro.PremiumProduct_PRESTO)
				So(err, ShouldNotBeNil)
				So(resp, ShouldBeNil)
			})
		})
	})
}

func TestTurboCode(t *testing.T) {
	Convey("Test TurboCode", t, func() {
		mockISubServiceClient := new(MockISubServiceClient)
		mockClients := clients.Clients{
			SubsServiceClient: mockISubServiceClient,
		}
		premiumStatuses := PremiumStatusesImpl{
			Clients: mockClients,
		}

		mockTurboSubscription1 := voyagertwirp.Subscription{
			ProductId:  "324",
			AccessGuid: "324AG",
		}

		mockTurboSubscription2 := voyagertwirp.Subscription{
			ProductId:  "388",
			AccessGuid: "388AG",
		}

		Convey("Test GetTurboCode", func() {
			Convey("returns turbo monthly access guid if a user has the turbo monthly subscription", func() {
				res := voyagertwirp.GetUserProductSubscriptionsResponse{
					Subscriptions: []*voyagertwirp.Subscription{&mockTurboSubscription1},
				}
				mockISubServiceClient.On("GetUserProductSubscriptions", mock.Anything, mock.Anything).Return(&res, nil)

				result, err := premiumStatuses.GetTurboCode("id")
				So(err, ShouldBeNil)
				So(result, ShouldEqual, mockTurboSubscription1.AccessGuid)
			})

			Convey("returns turbo yearly access guid if a user has the turbo yearly subscription", func() {
				res := voyagertwirp.GetUserProductSubscriptionsResponse{
					Subscriptions: []*voyagertwirp.Subscription{&mockTurboSubscription2},
				}
				mockISubServiceClient.On("GetUserProductSubscriptions", mock.Anything, mock.Anything).Return(&res, nil)

				result, err := premiumStatuses.GetTurboCode("id")
				So(err, ShouldBeNil)
				So(result, ShouldEqual, mockTurboSubscription2.AccessGuid)
			})

			Convey("returns emptyString if a user has no subscriptions", func() {
				res := voyagertwirp.GetUserProductSubscriptionsResponse{
					Subscriptions: []*voyagertwirp.Subscription{},
				}
				mockISubServiceClient.On("GetUserProductSubscriptions", mock.Anything, mock.Anything).Return(&res, nil)

				result, err := premiumStatuses.GetTurboCode("id")
				So(err, ShouldBeNil)
				So(result, ShouldEqual, "")
			})

			Convey("returns error when subs service has an error", func() {
				mockISubServiceClient.On("GetUserProductSubscriptions", mock.Anything, mock.Anything).Return(nil, errors.New("Error calling subs service"))

				_, err := premiumStatuses.GetTurboCode("id")
				So(err, ShouldNotBeNil)
			})
		})
	})
}
