package badges

import (
	"bytes"
	"context"
	"encoding/json"
	"net/http"
	"net/url"
	"time"

	"code.justin.tv/chat/badges/app/models"
	"code.justin.tv/foundation/twitchclient"
	"code.justin.tv/samus/nitro/config"
	"code.justin.tv/samus/nitro/metrics"
)

const (
	defaultStatSampleRate = 1.0
	badgesServiceName     = "Badges"
)

type IBadgesClient interface {
	// GrantPrimeBadge entitles a user to the "Prime" badge globally
	// Returns error if the operation was not successful
	GrantPrimeBadge(ctx context.Context, userID string) error
	// RemovePrimeBadge removes the "Prime" badge for a user globally
	// Returns error if the operation was not successful
	RemovePrimeBadge(ctx context.Context, userID string) error
}

type BadgesClient struct {
	twitchClient twitchclient.Client
	metricLogger metrics.IMetricLogger
}

func NewBadgesClient(config *config.Configuration, metricLogger metrics.IMetricLogger) (IBadgesClient, error) {
	twitchClient, err := twitchclient.NewClient(twitchclient.ClientConf{
		Host: config.BadgesHost,
	})
	if err != nil {
		return nil, err
	}

	return &BadgesClient{twitchClient, metricLogger}, nil
}

func (b *BadgesClient) GrantPrimeBadge(ctx context.Context, userID string) error {
	url := url.URL{Path: "/v1/badges/users/" + userID + "/global/prime"}

	body, err := json.Marshal(models.GrantBadgeParams{})
	if err != nil {
		return err
	}

	start := time.Now()
	req, err := b.twitchClient.NewRequest("POST", url.String(), bytes.NewReader(body))
	if err != nil {
		return err
	}

	reqOpts := twitchclient.ReqOpts{
		StatName:       "service.badges.grant_prime_badge",
		StatSampleRate: defaultStatSampleRate,
	}

	resp, err := b.twitchClient.Do(ctx, req, reqOpts)

	if err != nil {
		return err
	}

	b.metricLogger.AddHttpDependencyCallMetrics(resp.StatusCode, time.Since(start), badgesServiceName)
	if http.StatusOK != resp.StatusCode {
		return &twitchclient.Error{
			Message:    "Could not grant prime badge",
			StatusCode: resp.StatusCode,
		}
	}

	return nil
}

func (b *BadgesClient) RemovePrimeBadge(ctx context.Context, userID string) error {
	url := url.URL{Path: "/v1/badges/users/" + userID + "/global/prime"}
	body, err := json.Marshal(models.GrantBadgeParams{})
	if err != nil {
		return err
	}

	start := time.Now()
	req, err := b.twitchClient.NewRequest("DELETE", url.String(), bytes.NewReader(body))
	if err != nil {
		return err
	}

	reqOpts := twitchclient.ReqOpts{
		StatName:       "service.badges.remove_prime_badge",
		StatSampleRate: defaultStatSampleRate,
	}

	resp, err := b.twitchClient.Do(ctx, req, reqOpts)

	if err != nil {
		return err
	}

	b.metricLogger.AddHttpDependencyCallMetrics(resp.StatusCode, time.Since(start), badgesServiceName)
	if http.StatusOK != resp.StatusCode {
		return &twitchclient.Error{
			Message:    "Could not remove prime badge",
			StatusCode: resp.StatusCode,
		}
	}

	return nil
}
