package clients

import (
	voyagertwirp "code.justin.tv/amzn/TwitchVoyagerTwirp"
	"code.justin.tv/foundation/twitchclient"
	substwirp "code.justin.tv/revenue/subscriptions/twirp"
	"code.justin.tv/samus/nitro/config"
	"code.justin.tv/samus/nitro/internal/clients/badges"
	"code.justin.tv/samus/nitro/internal/clients/dynamo"
	"code.justin.tv/samus/nitro/internal/clients/mako"
	"code.justin.tv/samus/nitro/internal/clients/payments"
	"code.justin.tv/samus/nitro/internal/clients/subs"
	"code.justin.tv/samus/nitro/internal/clients/user_status"
	"code.justin.tv/samus/nitro/metrics"
	"github.com/pkg/errors"
)

type Clients struct {
	BadgesClient                     badges.IBadgesClient
	MakoClient                       mako.IMakoClient
	PaymentsClient                   payments.IPaymentsClient
	SubsServiceClient                subs.ISubServiceClient
	UserStatusClient                 user_status.IUserStatusClient
	AdditionalPremiumStatusDynamoDao dynamo.IAdditionalPremiumStatusDynamoDao
}

// SetupClients initializes all the clients
func SetupClients(cfg *config.Configuration, metricLogger metrics.IMetricLogger) (*Clients, error) {

	// Badges
	badgesClient, err := badges.NewBadgesClient(cfg, metricLogger)
	if err != nil {
		return nil, errors.Wrap(err, "Failed to create Badges client")
	}

	// Mako (Smilies)
	makoClient, err := mako.NewMakoClient(cfg, metricLogger)
	if err != nil {
		return nil, errors.Wrap(err, "Failed to create Mako client")
	}

	// Payments
	paymentsClient, err := payments.NewPaymentsClient(cfg, metricLogger)
	if err != nil {
		return nil, errors.Wrap(err, "Failed to create Payments client")
	}

	// Subs
	subscriptionsClient := substwirp.NewSubscriptionsProtobufClient(cfg.SubscriptionsHost, twitchclient.NewHTTPClient(twitchclient.ClientConf{
		Host: cfg.SubscriptionsHost,
		Transport: twitchclient.TransportConf{
			MaxIdleConnsPerHost: cfg.SubscriptionsMaxIdleConnections,
		},
	}))

	// Voyager
	voyagerClient := voyagertwirp.NewTwitchVoyagerProtobufClient(cfg.TwitchVoyagerHost, twitchclient.NewHTTPClient(twitchclient.ClientConf{
		Host: cfg.TwitchVoyagerHost,
		Transport: twitchclient.TransportConf{
			MaxIdleConnsPerHost: cfg.TwitchVoyagerMaxIdleConnections,
		},
	}))

	subscriptionsWrapper := subs.New(subscriptionsClient, voyagerClient, metricLogger)

	// UserStatus
	userStatusClient, err := user_status.NewUserStatusClient(cfg)
	if err != nil {
		return nil, errors.Wrap(err, "Failed to create UserStatusClient")
	}

	additionalPremiumStatusDynamoDaoClient := dynamo.NewClient(&dynamo.DynamoClientConfig{
		AwsRegion: cfg.AWSRegion,
		TableName: cfg.AdditionalPremiumStatusDynamo,
	})
	additionalPremiumStatusDynamoDao := dynamo.NewAdditionalPremiumStatusDynamoDao(additionalPremiumStatusDynamoDaoClient)

	return &Clients{
		BadgesClient:                     badgesClient,
		MakoClient:                       makoClient,
		PaymentsClient:                   paymentsClient,
		SubsServiceClient:                subscriptionsWrapper,
		UserStatusClient:                 userStatusClient,
		AdditionalPremiumStatusDynamoDao: additionalPremiumStatusDynamoDao,
	}, nil
}
