package dynamo

import (
	"fmt"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/credentials"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/pkg/errors"
	log "github.com/sirupsen/logrus"
)

type DynamoClientConfig struct {
	AwsRegion           string
	TableName           string
	EndpointOverride    string
	CredentialsOverride *credentials.Credentials
}

type DynamoClient struct {
	Dynamo       *dynamodb.DynamoDB
	ClientConfig *DynamoClientConfig
}

func NewClient(config *DynamoClientConfig) *DynamoClient {
	awsConfig := &aws.Config{
		Region:      aws.String(config.AwsRegion),
		Credentials: config.CredentialsOverride,
	}
	sess, err := session.NewSession()

	if err != nil {
		log.Fatalf("Failed to initialize the DB session: %s", err)
		return nil
	}
	dynamo := dynamodb.New(sess, awsConfig)
	if config.EndpointOverride != "" {
		dynamo.Endpoint = config.EndpointOverride
	}
	return &DynamoClient{
		Dynamo:       dynamo,
		ClientConfig: config,
	}
}

func (c *DynamoClient) GetItem(record DynamoTableRecord) (DynamoTableRecord, error) {
	tableName := c.ClientConfig.TableName
	result, err := c.Dynamo.GetItem(NewGetItemInput(tableName, record))
	if err != nil {
		return nil, errors.Wrap(err, "DynamoClient: Encountered an error calling dynamo GetItem")
	}

	// Result map is empty if no record is found
	if len(result.Item) != 0 {
		record, err := record.GetTable().ConvertAttributeMapToRecord(result.Item)
		if err != nil {
			return nil, errors.Wrap(err, "DynamoClient: Encountered an error converting dynamo GetItem result to record")
		}
		return record, nil
	} else {
		return nil, nil
	}
}

func (c *DynamoClient) PutItem(record DynamoTableRecord) error {
	tableName := c.ClientConfig.TableName
	_, err := c.Dynamo.PutItem(NewPutItemInput(tableName, record))
	if err != nil {
		log.Errorf("DynamoClient: Encountered an error calling dynamo PutItem, %v", err)
		return err
	}
	return nil
}

func (c *DynamoClient) DeleteItem(record DynamoTableRecord) error {
	tableName := c.ClientConfig.TableName
	_, err := c.Dynamo.DeleteItem(NewDeleteItemInput(tableName, record))
	if err != nil {
		msg := fmt.Sprintf("DynamoClient: Encountered an error calling dynamo DeleteItem")
		log.WithError(err).Error(msg)
		return errors.Wrap(err, msg)
	}
	return nil
}
