package mako

import (
	"time"

	"code.justin.tv/commerce/mako-client"
	"code.justin.tv/foundation/twitchclient"
	"code.justin.tv/samus/nitro/config"
	"code.justin.tv/samus/nitro/metrics"
	"github.com/pkg/errors"
	"golang.org/x/net/context"
)

const (
	makoServiceName = "Mako"
)

type IMakoClient interface {
	// ResetSmiliesSetForUser resets the emotes set of a user to default (ROBOTS)
	ResetSmiliesSetForUser(ctx context.Context, userID string) (*mako_client.SetSmiliesResponse, error)
}

type MakoClient struct {
	makoClient   mako_client.Mako
	metricLogger metrics.IMetricLogger
}

func NewMakoClient(config *config.Configuration, metricLogger metrics.IMetricLogger) (IMakoClient, error) {
	makoClient := mako_client.NewMakoProtobufClient(
		config.MakoHost,
		twitchclient.NewHTTPClient(twitchclient.ClientConf{Host: config.MakoHost}))

	return &MakoClient{makoClient, metricLogger}, nil
}

func (m *MakoClient) ResetSmiliesSetForUser(ctx context.Context, userID string) (*mako_client.SetSmiliesResponse, error) {
	start := time.Now()
	resp, err := m.makoClient.SetSmilies(ctx, &mako_client.SetSmiliesRequest{
		UserId:       userID,
		SmiliesSetId: mako_client.SmiliesSetId_ROBOTS,
	})

	if err != nil {
		return nil, errors.Wrap(err, "Unable to set smilies for user "+userID)
	}

	m.metricLogger.AddTwirpDependencyCallMetrics(err, time.Since(start), makoServiceName)

	return resp, nil
}
