package payments

import (
	"bytes"
	"context"
	"encoding/json"
	"net/url"

	"time"

	"code.justin.tv/foundation/twitchclient"
	"code.justin.tv/revenue/payments-service-go-client/client"
	"code.justin.tv/samus/nitro/config"
	"code.justin.tv/samus/nitro/metrics"
)

const (
	defaultStatSampleRate = 1.0
	paymentsServiceName   = "Payments"
)

type IPaymentsClient interface {
	CancelPurchase(ctx context.Context, userID string, productShortName string, request payments.CancelPurchaseRequest) error
	CompletePurchase(ctx context.Context, userID string, productShortName string, request payments.CompletePurchaseRequest) (*payments.PurchaseProfile, error)
}

type PaymentsClient struct {
	twitchClient   twitchclient.Client
	paymentsClient payments.Client
	metricLogger   metrics.IMetricLogger
}

func NewPaymentsClient(config *config.Configuration, metricLogger metrics.IMetricLogger) (IPaymentsClient, error) {
	cfg := twitchclient.ClientConf{
		Host: config.PaymentsHost,
	}

	twitchClient, err := twitchclient.NewClient(cfg)
	if err != nil {
		return nil, err
	}

	paymentsClient, err := payments.NewClient(cfg)
	if err != nil {
		return nil, err
	}

	return &PaymentsClient{twitchClient, paymentsClient, metricLogger}, nil
}

func (p *PaymentsClient) CompletePurchase(ctx context.Context, userID string, productShortName string, request payments.CompletePurchaseRequest) (*payments.PurchaseProfile, error) {
	url := url.URL{Path: "/internal/users/" + userID + "/products/" + productShortName + "/purchase/complete"}

	body, err := json.Marshal(request)
	if err != nil {
		return nil, err
	}

	req, err := p.twitchClient.NewRequest("POST", url.String(), bytes.NewBuffer(body))
	if err != nil {
		return nil, err
	}
	req.Header.Set("Content-Type", "application/json")

	reqOpts := twitchclient.ReqOpts{
		StatName:       "service.payments.complete_purchase",
		StatSampleRate: defaultStatSampleRate,
	}

	start := time.Now()
	resp, err := p.twitchClient.Do(ctx, req, reqOpts)

	if err != nil {
		return nil, err
	}
	p.metricLogger.AddHttpDependencyCallMetrics(resp.StatusCode, time.Since(start), paymentsServiceName)

	var purchaseProfile payments.PurchaseProfile
	err = json.NewDecoder(resp.Body).Decode(&purchaseProfile)
	if err != nil {
		return nil, err
	}

	return &purchaseProfile, nil
}

func (p *PaymentsClient) CancelPurchase(ctx context.Context, userID string, productShortName string, request payments.CancelPurchaseRequest) error {
	url := url.URL{Path: "/internal/users/" + userID + "/products/" + productShortName + "/purchase/cancel"}

	body, err := json.Marshal(request)
	if err != nil {
		return err
	}

	req, err := p.twitchClient.NewRequest("PUT", url.String(), bytes.NewBuffer(body))
	if err != nil {
		return err
	}
	req.Header.Set("Content-Type", "application/json")

	reqOpts := twitchclient.ReqOpts{
		StatName:       "service.payments.cancel_purchase",
		StatSampleRate: defaultStatSampleRate,
	}
	start := time.Now()
	resp, err := p.twitchClient.Do(ctx, req, reqOpts)
	p.metricLogger.AddHttpDependencyCallMetrics(resp.StatusCode, time.Since(start), paymentsServiceName)

	return err
}
