package payments

import (
	"bytes"
	"context"
	"encoding/json"
	"io/ioutil"
	"net/http"
	"testing"

	"code.justin.tv/revenue/payments-service-go-client/client"
	"code.justin.tv/revenue/subscriptions/twirp"
	"code.justin.tv/samus/nitro/_tools/src/github.com/pkg/errors"
	"code.justin.tv/samus/nitro/metrics"
	MockPaymentsClient "code.justin.tv/samus/nitro/mocks/PaymentsClient"
	MockTwitchClient "code.justin.tv/samus/nitro/mocks/TwitchClient"
	. "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"
)

func TestPaymentsClient(t *testing.T) {
	Convey("Testing Payments client APIs", t, func() {
		mockTwitchClient := new(MockTwitchClient.Client)
		mockPaymentsClient := new(MockPaymentsClient.Client)
		mockMetricLogger := new(metrics.MockIMetricLogger)

		mockTwitchClient.
			On("NewRequest", mock.Anything, mock.Anything, mock.Anything).
			Return(http.NewRequest("", "mocked/path", nil))

		paymentsClient := PaymentsClient{mockTwitchClient, mockPaymentsClient, mockMetricLogger}

		Convey("Testing CompletePurchase client API", func() {
			userID := "fake-user-id"
			productShortName := "fake-product-short-name"

			request := payments.CompletePurchaseRequest{
				PaymentProvider: "fake-provider",
				PaymentInfo: payments.PaymentInfo{
					FirstName: "fake-first-name", LastName: "fake-last-name",
					Token: "fake-token", Gateway: "fake-gateway",
					State: "fake-state", Country: "fake-country",
				},
				Platform: "fake-platform",
			}

			Convey("with success", func() {
				purchaseProfile := payments.PurchaseProfile{}
				respBody, err := json.Marshal(purchaseProfile)
				if err != nil {
					t.Errorf("Error: %+v", err)
					t.Fail()
				}

				mockResp := new(http.Response)
				mockResp.StatusCode = http.StatusOK
				mockResp.Body = ioutil.NopCloser(bytes.NewBufferString(string(respBody[:])))

				mockTwitchClient.On("Do", context.TODO(), mock.Anything, mock.Anything).Return(mockResp, nil)
				mockMetricLogger.On("AddHttpDependencyCallMetrics", mock.Anything, mock.Anything, mock.Anything).Return()

				resp, err := paymentsClient.CompletePurchase(context.TODO(), userID, productShortName, request)
				So(err, ShouldBeNil)
				So(resp, ShouldNotBeNil)
			})

			Convey("with failure", func() {
				mockResp := new(http.Response)
				mockResp.StatusCode = http.StatusServiceUnavailable

				mockTwitchClient.On("Do", context.TODO(), mock.Anything, mock.Anything).Return(mockResp, errors.New("Oops, something's not right..."))
				mockMetricLogger.On("AddHttpDependencyCallMetrics", mock.Anything, mock.Anything, mock.Anything).Return()

				resp, err := paymentsClient.CompletePurchase(context.TODO(), userID, productShortName, request)
				So(err, ShouldNotBeNil)
				So(resp, ShouldBeNil)
			})
		})

		Convey("Testing CancelPurchase client API", func() {
			userID := "fake-user-id"
			productShortName := "fake-product-short-name"

			subscription := substwirp.Subscription{
				OriginId: "fake-origin-id",
			}

			request := payments.CancelPurchaseRequest{
				PurchaseProfileID: subscription.OriginId,
			}

			Convey("with success", func() {
				mockResp := new(http.Response)
				mockResp.StatusCode = http.StatusOK

				mockTwitchClient.On("Do", context.TODO(), mock.Anything, mock.Anything).Return(mockResp, nil)
				mockMetricLogger.On("AddHttpDependencyCallMetrics", mock.Anything, mock.Anything, mock.Anything).Return()

				err := paymentsClient.CancelPurchase(context.TODO(), userID, productShortName, request)
				So(err, ShouldBeNil)
			})

			Convey("with failure", func() {
				mockResp := new(http.Response)
				mockResp.StatusCode = http.StatusInternalServerError
				mockTwitchClient.On("Do", context.TODO(), mock.Anything, mock.Anything).Return(mockResp, errors.New("Oops, something's not right..."))
				mockMetricLogger.On("AddHttpDependencyCallMetrics", mock.Anything, mock.Anything, mock.Anything).Return()

				err := paymentsClient.CancelPurchase(context.TODO(), userID, productShortName, payments.CancelPurchaseRequest{})
				So(err, ShouldNotBeNil)
			})
		})
	})
}
