package subs

import (
	"time"

	voyagertwirp "code.justin.tv/amzn/TwitchVoyagerTwirp"
	substwirp "code.justin.tv/revenue/subscriptions/twirp"
	"code.justin.tv/samus/nitro/_tools/src/github.com/pkg/errors"
	"code.justin.tv/samus/nitro/internal/clients/wrapper"
	"code.justin.tv/samus/nitro/metrics"
	"github.com/golang/protobuf/ptypes"
)

const (
	subscriptionsServiceName           = "Subs"
	getUserProductSubscriptionsAPIName = "GetUserProductSubscriptions"
	updateTicketEndDateAPIName         = "UpdateTicketEndDate"
)

type ISubServiceClient interface {
	GetUserProductSubscriptions(twitchUserID string, productIDS []string) (*voyagertwirp.GetUserProductSubscriptionsResponse, error)
	UpdateTicketEndDate(ticketID string, endDate time.Time) error
}

type SubsServiceImpl struct {
	SubsClient    substwirp.Subscriptions
	VoyagerClient voyagertwirp.TwitchVoyager
	metricLogger  metrics.IMetricLogger
}

// New creates an implementation of the ISubServiceClient interface
func New(c substwirp.Subscriptions, v voyagertwirp.TwitchVoyager, m metrics.IMetricLogger) ISubServiceClient {
	return &SubsServiceImpl{
		SubsClient:    c,
		VoyagerClient: v,
		metricLogger:  m,
	}
}

// GetUserProductSubscriptions API retrieves a Twitch user's product subscriptions given the twitchUserID
func (s *SubsServiceImpl) GetUserProductSubscriptions(twitchUserID string, productIDS []string) (*voyagertwirp.GetUserProductSubscriptionsResponse, error) {
	req := &voyagertwirp.GetUserProductSubscriptionsRequest{
		UserId:     twitchUserID,
		ProductIds: productIDS,
	}
	ctx := wrapper.GenerateRequestContext(subscriptionsServiceName, getUserProductSubscriptionsAPIName)

	start := time.Now()
	res, err := s.VoyagerClient.GetUserProductSubscriptions(ctx, req)
	s.metricLogger.AddTwirpDependencyCallMetrics(err, time.Since(start), subscriptionsServiceName)

	if err != nil {
		return nil, errors.Wrap(err, "GetUserProductSubscriptions call failed")
	}
	return res, nil
}

func (s *SubsServiceImpl) UpdateTicketEndDate(ticketID string, endDate time.Time) error {
	endTimestamp, err := ptypes.TimestampProto(endDate)
	if err != nil {
		return errors.Wrap(err, "UpdateTicketEndDate call failed")
	}

	ctx := wrapper.GenerateRequestContext(subscriptionsServiceName, updateTicketEndDateAPIName)

	req := &substwirp.UpdateSubscriptionRequest{
		Subscription: &substwirp.UpdateSubscriptionRequest_Id{
			Id: ticketID,
		},
		BenefitEnd: endTimestamp,
	}

	start := time.Now()
	_, err = s.SubsClient.UpdateSubscription(ctx, req)
	s.metricLogger.AddTwirpDependencyCallMetrics(err, time.Since(start), subscriptionsServiceName)
	if err != nil {
		return errors.Wrap(err, "UpdateSubscription call failed")
	}

	return nil
}
