package user_status

import (
	"time"

	"context"

	"code.justin.tv/samus/nitro/_tools/src/github.com/pkg/errors"
	"code.justin.tv/samus/nitro/config"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/guregu/dynamo"
)

const (
	TwitchUserID           = "twitchUserId"
	GrandfatheringFlagName = "isAcceptingUsers"
	FlagName               = "flagName"
	Status                 = "status"
)

type GrandfatheredUser struct {
	TwitchUserID            string    `dynamo:"twitchUserId"`
	HasOldTwitchPrime       bool      `dynamo:"hasOldTwitchPrime"`
	HasNewTwitchPrime       bool      `dynamo:"hasNewTwitchPrime"`
	EndOfOldTwitchPrimeDate time.Time `dynamo:"endOfOldTwitchPrimeDate"`
}

type Flag struct {
	FlagName string `dynamo:"flagName"`
	Status   bool   `dynamo:"status"`
}

type IUserStatusClient interface {

	// Retrieve a Grandfathered user given a twitchUserID.
	GetGrandfatheredUser(string) (*GrandfatheredUser, error)

	// Create a new GrandfatheredUser entry. Will not overwrite.
	CreateGrandfatheredUser(*GrandfatheredUser) error

	// Update an existing GrandfatheredUser entry.
	UpdateGrandfatheredUser(*GrandfatheredUser) error

	// Get flag (hardcoded key)
	GetGrandfatheringFlag() (*Flag, error)

	// Delete an existing GrandfatheredUser entry.
	DeleteGrandfatheredUser(string) error

	// Put a GrandfatheredUser IRREGARDLESS (yes, irregardless) of whether the user exists or not.
	PutGrandfatheredUser(*GrandfatheredUser) error
}

type UserStatusClient struct {
	grandfatheredUsersTable  *dynamo.Table
	grandfatheringFlagsTable *dynamo.Table
}

func NewUserStatusClient(config *config.Configuration) (IUserStatusClient, error) {
	db := dynamo.New(session.New(), &aws.Config{Region: aws.String(config.AWSRegion)})
	grandfatheredUsersTable := db.Table(config.GrandfatheredUsersTable)
	grandfatheringFlagsTable := db.Table(config.GrandfatheringFlagsTable)
	return &UserStatusClient{
		grandfatheredUsersTable:  &grandfatheredUsersTable,
		grandfatheringFlagsTable: &grandfatheringFlagsTable,
	}, nil
}

func (s *UserStatusClient) GetGrandfatheredUser(twitchUserID string) (*GrandfatheredUser, error) {
	var user GrandfatheredUser
	ctx, cancel := context.WithTimeout(context.Background(), 500*time.Millisecond)
	defer cancel()
	err := s.grandfatheredUsersTable.Get(TwitchUserID, twitchUserID).OneWithContext(ctx, &user)

	if err == dynamo.ErrNotFound {
		return nil, nil
	}

	return &user, errors.Wrap(err, "UserStatusClient.GetGrandfatheredUser")
}

func (s *UserStatusClient) PutGrandfatheredUser(user *GrandfatheredUser) error {
	ctx, cancel := context.WithTimeout(context.Background(), 500*time.Millisecond)
	defer cancel()
	err := s.grandfatheredUsersTable.
		Put(user).RunWithContext(ctx)

	// errors.Wrap returns nil if <error> argument is nil.
	return errors.Wrap(err, "UserStatusClient.CreateGrandfatheredUser")
}

func (s *UserStatusClient) CreateGrandfatheredUser(grandfatheredUser *GrandfatheredUser) error {
	ctx, cancel := context.WithTimeout(context.Background(), 500*time.Millisecond)
	defer cancel()
	err := s.grandfatheredUsersTable.
		Put(grandfatheredUser).
		If("attribute_not_exists($)", TwitchUserID).
		RunWithContext(ctx)

	// errors.Wrap returns nil if <error> argument is nil.
	return errors.Wrap(err, "UserStatusClient.CreateGrandfatheredUser")
}

func (s *UserStatusClient) UpdateGrandfatheredUser(grandfatheredUser *GrandfatheredUser) error {
	ctx, cancel := context.WithTimeout(context.Background(), 500*time.Millisecond)
	defer cancel()
	err := s.grandfatheredUsersTable.
		Put(grandfatheredUser).
		If("attribute_exists($)", TwitchUserID).
		RunWithContext(ctx)

	// errors.Wrap returns nil if <error> argument is nil.
	return errors.Wrap(err, "UserStatusClient.UpdateGrandfatheredUser")
}

func (s *UserStatusClient) GetGrandfatheringFlag() (*Flag, error) {
	var isGrandfatheringUsersFlag Flag
	ctx, cancel := context.WithTimeout(context.Background(), 500*time.Millisecond)
	defer cancel()
	err := s.grandfatheringFlagsTable.
		Get(FlagName, GrandfatheringFlagName).
		OneWithContext(ctx, &isGrandfatheringUsersFlag)

	// errors.Wrap returns nil if <error> argument is nil.
	return &isGrandfatheringUsersFlag, errors.Wrap(err, "UserStatusClient.GetGrandfatheringFlag")
}

func (s *UserStatusClient) DeleteGrandfatheredUser(twitchUserID string) error {
	ctx, cancel := context.WithTimeout(context.Background(), 500*time.Millisecond)
	defer cancel()
	err := s.grandfatheredUsersTable.
		Delete(TwitchUserID, twitchUserID).
		RunWithContext(ctx)

	return errors.Wrap(err, "UserStatusClient.DeleteGrandfatheredUser")
}
