package middleware

import (
	"bytes"
	"io"
	"io/ioutil"
	"net/http"

	log "github.com/sirupsen/logrus"
)

// ResponseSaver saves the response body so we can log it
type ResponseSaver struct {
	responseBody *bytes.Buffer
	statusCode   int
	http.ResponseWriter
}

func (r *ResponseSaver) Write(body []byte) (int, error) {
	return r.responseBody.Write(body)
}

func (r *ResponseSaver) WriteHeader(code int) {
	r.statusCode = code
	r.ResponseWriter.WriteHeader(code)
}

// AccessLogger logs the http request, response and timestamp (including all headers)
func AccessLogger(h http.Handler) http.Handler {
	return http.HandlerFunc(func(responseWriter http.ResponseWriter, request *http.Request) {
		// Read the http request body
		requestBody, err := ioutil.ReadAll(request.Body)
		if err != nil {
			log.WithFields(log.Fields{
				"Request": request,
			}).Error("Received an invalid request")
			return
		}
		request.Body = ioutil.NopCloser(bytes.NewBuffer(requestBody))

		responseSaver := &ResponseSaver{
			responseBody:   &bytes.Buffer{},
			statusCode:     http.StatusOK, // must default to OK to handle implicit OK responses
			ResponseWriter: responseWriter,
		}
		h.ServeHTTP(responseSaver, request)

		log.WithFields(log.Fields{
			"Method":       request.Method,
			"RequestBody":  string(requestBody),
			"ResponseBody": responseSaver.responseBody,
			"StatusCode":   responseSaver.statusCode,
			"URL":          request.URL.Path,
		}).Info()

		// Copy out our response
		_, err = io.Copy(responseWriter, responseSaver.responseBody)
		if err != nil {
			log.WithFields(log.Fields{
				"Request":  request,
				"Response": responseSaver.responseBody,
			}).Error("Error creating a response")
			return
		}
	})
}
