package middleware

import (
	"fmt"
	"net/http"
	"strings"

	"code.justin.tv/samus/nitro/metrics"
	"code.justin.tv/samus/nitro/rpc"
	"github.com/felixge/httpsnoop"
)

type MetricsMiddleware struct {
	metricLogger metrics.IMetricLogger
}

// NewMetricsMiddleware provides the Logger to capture API metrics on Nitro
func NewMetricsMiddleware(metricLogger metrics.IMetricLogger) *MetricsMiddleware {
	return &MetricsMiddleware{
		metricLogger: metricLogger,
	}
}

func isTwirpRequest(r *http.Request) bool {
	if r == nil || r.URL == nil {
		return false
	}
	return strings.HasPrefix(r.URL.Path, nitro.NitroPathPrefix)
}

func getTwirpMetricName(r *http.Request) string {
	if r == nil || r.URL == nil {
		return ""
	}
	return fmt.Sprintf("TWIRP:%s", strings.TrimPrefix(r.URL.Path, nitro.NitroPathPrefix))
}

// Metrics middleware method to get API level metrics (taken from paydays cloudwatch metrics)
func (m *MetricsMiddleware) Metrics(next http.Handler) http.Handler {
	fn := func(w http.ResponseWriter, r *http.Request) {
		metrics := httpsnoop.CaptureMetrics(next, w, r)

		var metricName string
		if isTwirpRequest(r) {
			metricName = getTwirpMetricName(r)
		} else {
			metricName = fmt.Sprintf("%s:%s", r.Method, r.URL)
		}

		m.metricLogger.LogDurationMetric(metricName, metrics.Duration)
		m.metricLogger.LogCountMetricsForStatusCode(metrics.Code, metricName)
	}
	return http.HandlerFunc(fn)
}
