package gifting

import (
	"code.justin.tv/samus/rex/config"
	"golang.org/x/net/context"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/awserr"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/guregu/dynamo"
	"github.com/satori/go.uuid"
)

const (
	DynamoDBGiftTableName          = "Gifts"
	DynamoDBClaimableGiftTableName = "ClaimableGifts"

	DynamoDBTUIDColumn    = "tuid"
	DynamoDBOfferIdColumn = "offerId"
	DynamoDBGiftIDColumn  = "giftId"
	DynamoDBFromIndex     = "from-index"
	DynamoDBToIndex       = "to-index"
	DynamoDBStatusColumn  = "status"

	DynamoDBFromColumn = "from"
	DynamoDBToColumn   = "to"

	DynamoDBOfferFilter           = "offerId = ?"
	DynamoDBGiftIDFiler           = "giftId = ?"
	DynamoDBClaimableGifterFilter = "attribute_not_exists(tuid) AND attribute_not_exists(offerId)"

	GiftStatusOnCreation = "GIFT_CLAIM_PENDING"
	GiftStatusOnClaim    = "GIFT_CLAIMED"
)

type Gift struct {
	GiftID  string
	From    string
	OfferID string
	To      string
	Status  string
}

func (g Gift) IsClaimable() bool {
	return g.Status == GiftStatusOnCreation
}

type ClaimableGift struct {
	Tuid    string
	OfferID string
	GiftID  string
}

type DynamoGift struct {
	GiftID  string `dynamo:"giftId"`
	From    string `dynamo:"from"`
	OfferID string `dynamo:"offerId"`
	To      string `dynamo:"to"`
	Status  string `dynamo:"status"`
}

type DynamoClaimableGift struct {
	Tuid    string `dynamo:"tuid"`
	OfferID string `dynamo:"offerId"`
	GiftID  string `dynamo:"giftId"`
}

type IGiftingClient interface {
	GetGift(ctx context.Context, giftID string) (*Gift, error)
	ClaimGift(ctx context.Context, giftID string) (*Gift, error)
	CreateGift(ctx context.Context, fromID string, offerID string, toID string) (*Gift, error)
	GetOrCreateClaimableGift(ctx context.Context, gift *Gift) (*ClaimableGift, error)
	GetClaimableGift(ctx context.Context, tuid string, offerID string) (*ClaimableGift, error)
	GetGiftsFrom(ctx context.Context, userID string) (*[]Gift, error)
	GetGiftsFromForOffer(ctx context.Context, userID string, offerID string) (*[]Gift, error)
	GetGiftsTo(ctx context.Context, userID string) (*[]Gift, error)
	GetGiftsToForOffer(ctx context.Context, userID string, offerID string) (*[]Gift, error)
}

type GiftingClient struct {
	giftTable          dynamo.Table
	claimableGiftTable dynamo.Table
}

func NewGiftingClient(config *config.Configuration) (IGiftingClient, error) {
	db := dynamo.New(session.New(), &aws.Config{Region: aws.String(config.AWSRegion)})
	giftTable := db.Table(DynamoDBGiftTableName)
	claimableGiftTable := db.Table(DynamoDBClaimableGiftTableName)
	return &GiftingClient{
		giftTable:          giftTable,
		claimableGiftTable: claimableGiftTable,
	}, nil
}

func isConditionalCheckErr(err error) bool {
	if ae, ok := err.(awserr.RequestFailure); ok {
		return ae.Code() == "ConditionalCheckFailedException"
	}
	return false
}

func convertToGift(gift *DynamoGift) *Gift {
	if gift == nil {
		return nil
	}

	return &Gift{
		GiftID:  gift.GiftID,
		From:    gift.From,
		OfferID: gift.OfferID,
		To:      gift.To,
		Status:  gift.Status,
	}
}

func convertToClaimableGift(claimableGift *DynamoClaimableGift) *ClaimableGift {
	if claimableGift == nil {
		return nil
	}

	return &ClaimableGift{
		Tuid:    claimableGift.Tuid,
		OfferID: claimableGift.OfferID,
		GiftID:  claimableGift.GiftID,
	}
}

func createDynamoGift(fromID string, offerID string, toID string) *DynamoGift {
	return &DynamoGift{
		GiftID:  uuid.NewV4().String(),
		From:    fromID,
		OfferID: offerID,
		To:      toID,
		Status:  GiftStatusOnCreation,
	}
}

func createDynamoClaimableGift(tuid string, offerID string, giftID string) *DynamoClaimableGift {
	return &DynamoClaimableGift{
		Tuid:    tuid,
		OfferID: offerID,
		GiftID:  giftID,
	}
}

func (s *GiftingClient) CreateGift(ctx context.Context, fromID string, offerID string, toID string) (*Gift, error) {
	gift := createDynamoGift(fromID, offerID, toID)
	err := s.giftTable.Put(gift).Run()
	return convertToGift(gift), err
}

func (s *GiftingClient) GetOrCreateClaimableGift(ctx context.Context, gift *Gift) (*ClaimableGift, error) {
	claimableGift := createDynamoClaimableGift(gift.To, gift.OfferID, gift.GiftID)
	err := s.claimableGiftTable.Put(claimableGift).If(DynamoDBClaimableGifterFilter).Run()

	if isConditionalCheckErr(err) {
		// Claimable gift already exists, so simply return it
		return s.GetClaimableGift(ctx, gift.To, gift.OfferID)
	}

	return convertToClaimableGift(claimableGift), err
}

func (s *GiftingClient) GetClaimableGift(ctx context.Context, tuid string, offerID string) (*ClaimableGift, error) {
	var result DynamoClaimableGift
	query := s.claimableGiftTable.Get(DynamoDBTUIDColumn, tuid).Range(DynamoDBOfferIdColumn, dynamo.Equal, offerID)
	err := query.One(&result)

	if err == dynamo.ErrNotFound {
		// If no item was found; its not an error; we just return nil.
		return nil, nil
	}

	return convertToClaimableGift(&result), err
}

func (s *GiftingClient) GetGift(ctx context.Context, giftID string) (*Gift, error) {
	var result DynamoGift
	query := s.giftTable.Get(DynamoDBGiftIDColumn, giftID)
	err := query.One(&result)

	if err != nil {
		return nil, err
	}

	return convertToGift(&result), err
}

func (s *GiftingClient) ClaimGift(ctx context.Context, giftID string) (*Gift, error) {
	var result DynamoGift
	err := s.giftTable.Update(DynamoDBGiftIDColumn, giftID).Set(DynamoDBStatusColumn, GiftStatusOnClaim).If(DynamoDBGiftIDFiler, giftID).Value(&result)

	return convertToGift(&result), err
}

func (s *GiftingClient) GetGiftsFromForOffer(ctx context.Context, userID string, offerID string) (*[]Gift, error) {
	return s.getGiftsFrom(ctx, userID, offerID)
}

func (s *GiftingClient) GetGiftsFrom(ctx context.Context, userID string) (*[]Gift, error) {
	return s.getGiftsFrom(ctx, userID, "")
}

func (s *GiftingClient) getGiftsFrom(ctx context.Context, userID string, offerID string) (*[]Gift, error) {
	var results []DynamoGift
	resultGifts := make([]Gift, 0)
	query := s.giftTable.Get(DynamoDBFromColumn, userID).Index(DynamoDBFromIndex)

	if len(offerID) > 0 {
		query = query.Filter(DynamoDBOfferFilter, offerID)
	}

	err := query.All(&results)

	for i := range results {
		convertedGift := convertToGift(&results[i])
		resultGifts = append(resultGifts, *convertedGift)
	}

	return &resultGifts, err
}

func (s *GiftingClient) GetGiftsToForOffer(ctx context.Context, userID string, offerID string) (*[]Gift, error) {
	return s.getGiftsTo(ctx, userID, offerID)
}

func (s *GiftingClient) GetGiftsTo(ctx context.Context, userID string) (*[]Gift, error) {
	return s.getGiftsTo(ctx, userID, "")
}

func (s *GiftingClient) getGiftsTo(ctx context.Context, userID string, offerID string) (*[]Gift, error) {
	var results []DynamoGift
	resultGifts := make([]Gift, 0)
	query := s.giftTable.Get(DynamoDBToColumn, userID).Index(DynamoDBToIndex)

	if len(offerID) > 0 {
		query = query.Filter(DynamoDBOfferFilter, offerID)
	}

	err := query.All(&results)

	for i := range results {
		convertedGift := convertToGift(&results[i])
		resultGifts = append(resultGifts, *convertedGift)
	}

	return &resultGifts, err
}
