package samus_offer_status

import (
	"encoding/json"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/aws/credentials"
	"github.com/aws/aws-sdk-go/aws/credentials/stscreds"
	"github.com/aws/aws-sdk-go/service/lambda"
	"code.justin.tv/common/config"
	"code.justin.tv/foundation/twitchclient"
	"github.com/aws/aws-sdk-go/service/sts"
	"golang.org/x/net/context"
	"time"
)

const (
	defaultStatSampleRate = 0.1
	defaultTimingXactName = "samus-offer-status"
)

type ErrorResponse struct {
	Status  int
	Message string
	Error   string
}

type Client interface {
	GetOfferStatus(ctx context.Context, userID string, offerIDs []string, reqOptsIn *twitchclient.ReqOpts) (*PrimeOfferStatusResponse, error)
	UpdateOfferStatus(ctx context.Context, userID string, offerStatuses []OfferAndStatusInput, reqOptsIn *twitchclient.ReqOpts) (*PrimeOfferStatusResponse, error)
}

type client struct {
	twitchClient   twitchclient.Client
	lambdaClient   *lambda.Lambda
	updateFunction string
	getFunction    string
}

var (
	prodGetFunction    		   = "arn:aws:lambda:us-west-2:869016631111:function:SamusOfferTrackingService-GetOffersStatusesForUser-11XAOKJZO5IKC"
	prodUpdateFunction 		   = "arn:aws:lambda:us-west-2:869016631111:function:SamusOfferTrackingService-UpdateOffersStatusesForU-N9V3JM7HYZFL"
	prodRoleArn        		   = "arn:aws:iam::869016631111:role/VisageLambdaRole-prod"

	developmentGetFunction 	   = "arn:aws:lambda:us-west-2:221636190748:function:SamusOfferTrackingService-GetOffersStatusesForUser-D3VRAWUE699W"
	developmentUpdateFunction  = "arn:aws:lambda:us-west-2:221636190748:function:SamusOfferTrackingService-UpdateOffersStatusesForU-RHAW489EET1U"
	developmentRoleArn		   = "arn:aws:iam::221636190748:role/VisageLambdaRole-beta"

	region             		   = "us-west-2"
)

func NewClient(conf twitchclient.ClientConf) (Client, error) {
	if conf.TimingXactName == "" {
		conf.TimingXactName = defaultTimingXactName
	}

	var roleArn, updateFunction, getFunction string
	switch environment := config.Environment(); environment {
	case "development": fallthrough
	case "staging":
		roleArn = developmentRoleArn
		getFunction = developmentGetFunction
		updateFunction = developmentUpdateFunction
	default:
		roleArn = prodRoleArn
		getFunction = prodGetFunction
		updateFunction = prodUpdateFunction
	}

	twitchClient, err := twitchclient.NewClient(conf)
	awsConfig := &aws.Config{Region: aws.String(region)}

	stsClient := sts.New(session.New(awsConfig))
	arp := &stscreds.AssumeRoleProvider{
		Duration:     3600 * time.Second, // 1 hour
		ExpiryWindow: 10 * time.Second,
		RoleARN:      roleArn,
		Client:       stsClient,
	}
	creds := credentials.NewCredentials(arp)

	// Create Coral Lambda client
	httpClient := twitchclient.NewHTTPClient(conf)
	config := aws.NewConfig().WithCredentials(creds).WithRegion(region).WithHTTPClient(httpClient).WithCredentialsChainVerboseErrors(true)

	svc := lambda.New(session.New(config))
	return &client{
		twitchClient: twitchClient,
		lambdaClient: svc,
		getFunction: getFunction,
		updateFunction: updateFunction,
	}, err
}

func (c *client) GetOfferStatus(ctx context.Context, userID string, offerIDs []string, reqOptsIn *twitchclient.ReqOpts) (*PrimeOfferStatusResponse, error) {
	body, err := json.Marshal(PrimeOfferStatusRequest{
		TwitchID: userID,
		OfferStatuses: offerIDs,
	})

	input := &lambda.InvokeInput{
		FunctionName: aws.String(c.getFunction),
		Payload:      []byte(body),
	}

	var response *lambda.InvokeOutput
	var responseError error
	if ctx != nil {
		response, responseError = c.lambdaClient.InvokeWithContext(ctx, input)
	} else {
		response, responseError = c.lambdaClient.Invoke(input)
	}

	if responseError != nil {
		return nil, responseError
	}

	var primeOfferStatusResponse PrimeOfferStatusResponse
	err = json.Unmarshal(response.Payload, &primeOfferStatusResponse)

	if err != nil {
		return nil, err
	}

	return &primeOfferStatusResponse, nil
}

func (c *client) UpdateOfferStatus(ctx context.Context, userID string, offerStatuses []OfferAndStatusInput, reqOptsIn *twitchclient.ReqOpts) (*PrimeOfferStatusResponse, error) {
	var offerStatusMap = make(map[string]string)
	for _, element := range offerStatuses {
		offerStatusMap[element.OfferID] = element.Status
	}

	body, err := json.Marshal(PrimeUpdateOfferStatusRequest{
		TwitchID: userID,
		OfferStatuses: offerStatusMap,
	})

	if err != nil {
		return nil, err
	}

	input := &lambda.InvokeInput{
		FunctionName: aws.String(c.updateFunction),
		Payload:      []byte(body),
	}

	var response *lambda.InvokeOutput
	var responseError error
	if ctx != nil {
		response, responseError = c.lambdaClient.InvokeWithContext(ctx, input)
	} else {
		response, responseError = c.lambdaClient.Invoke(input)
	}

	if responseError != nil {
		return nil, responseError
	}

	var primeOfferStatusResponse PrimeOfferStatusResponse
	err = json.Unmarshal(response.Payload, &primeOfferStatusResponse)

	return &primeOfferStatusResponse, nil
}