import fetchMiddleware from "core/store/middleware/fetch";

describe("Redux fetch middleware", () => {

  function setUp() {
    const store = {
      dispatch: jest.fn(),
      getState: jest.fn(() => ({})),
    };
    const next = jest.fn();

    const invoke = (action) => fetchMiddleware(store)(next)(action);

    return { store, next, invoke };
  }

  it("allows actions with non-Response type payloads to pass through.", () => {
    const { store, next, invoke } = setUp();
    const action = { type: "TEST" };

    // Critical statement.
    invoke(action);

    expect(store.dispatch).toHaveBeenCalledTimes(0);
    expect(next).toHaveBeenCalledWith(action);
  });

  it("reads the Response stream of actions with OK Response payloads.", () => {
    const { store, next, invoke } = setUp();
    const testType = "TEST";
    const testResponse = "tst";

    const initialAction = {
      payload: {
        json: jest.fn(() => testResponse),
        ok: true,
      },
      type: testType,
    };
    const expectedSubsequentAction = { type: testType, payload: testResponse };

    // Critical statement.
    invoke(initialAction);

    expect(store.dispatch).toHaveBeenCalledWith(expectedSubsequentAction);
    expect(initialAction.payload.json).toHaveBeenCalled();
    expect(next).toHaveBeenCalledWith(initialAction);
  });

  it("uses the Response stream to make an error with non-OK Response payloads", async () => {
    const { store, next, invoke } = setUp();
    const testType = "TEST";
    const testErrorCode = "NO_BALANCE";
    const testResponse = {
      errorCode: testErrorCode,
    };
    const jsonFunction = jest.fn(() => Promise.resolve(testResponse));

    const initialAction = {
      payload: {
        json: jsonFunction,
        ok: false,
      },
      type: testType,
    };
    const expectedSubsequentAction = {
      error: true,
      payload: new Error(testErrorCode),
      type: testType,
    };

    // Critical statement.
    invoke(initialAction);
    await jsonFunction;

    expect(store.dispatch).toHaveBeenCalledTimes(1);
    expect(store.dispatch).toHaveBeenCalledWith(expectedSubsequentAction);
    expect(initialAction.payload.json).toHaveBeenCalledTimes(1);
    expect(next).toHaveBeenCalledWith(initialAction);
  });

  it("dispatches a generic error with non-OK Response payloads if there is no errorCode", async () => {
    const { store, next, invoke } = setUp();
    const testType = "TEST";
    const testErrorCode = "NO_BALANCE";
    const jsonFunction = jest.fn(() => Promise.resolve({}));

    const initialAction = {
      payload: {
        json: jsonFunction,
        ok: false,
      },
      type: testType,
    };
    const expectedSubsequentAction = {
      error: true,
      payload: new Error("network error"),
      type: testType,
    };

    // Critical statement.
    invoke(initialAction);
    await jsonFunction;

    expect(store.dispatch).toHaveBeenCalledTimes(1);
    expect(store.dispatch).toHaveBeenCalledWith(expectedSubsequentAction);
    expect(initialAction.payload.json).toHaveBeenCalledTimes(1);
    expect(next).toHaveBeenCalledWith(initialAction);
  });

});
