import { configureMetricsMiddleware } from "core/store/middleware/metrics";
import { Location } from 'core/types/CommunityService'
import { buildReporter } from "core/utils/reporter";
import { MetricActions } from "core/store/middleware/metrics"

describe("Redux metrics middleware", () => {

  async function create() {
    const store = {
      dispatch: jest.fn(),
      getState: jest.fn(() => ({})),
    };
    const next = jest.fn();

    const reporter = { reportEvent: jest.fn() };
    const metricsMiddleware = configureMetricsMiddleware(reporter);

    const invoke = (action) => metricsMiddleware(store)(next)(action);

    return { store, next, invoke, reporter };
  }

  it("passes non-REPORT type actions through.", async () => {
    const { store, next, invoke } = await create();
    const action = { type: "TEST" };

    // Critical statement.
    invoke(action);

    expect(store.dispatch).toHaveBeenCalledTimes(0);
    expect(next).toHaveBeenCalledWith(action);
  });

  it("reports REPORT-type actions", async () => {
    const { store, next, invoke, reporter } = await create();
    store.getState = jest.fn(() => ({
      CommunityService: {
        location: Location.TOP_RIGHT,
        user: {
          credit: 1,
          hasPrime: true,
          twitchUserId: "diego",
        },
      },
      TwitchExtension: {
        authorization: {
          channelId: "channelid",
          userId: "userId",
        },
        context: {
          game: "game",
          mode: "mode",
        },
      },
      DISPLAY: {
        typeOfExtension: "overlay",
      },
    }));
    const extensionAction = { type: MetricActions.REPORT_EXTENSION_ACTION, payload: "action" };
    const primeReferralAction = { type: MetricActions.REPORT_TWITCH_PRIME_REFERRAL, payload: "referralID" };

    // Critical statement.
    invoke(extensionAction);

    expect(reporter.reportEvent).toHaveBeenCalledTimes(1);

    // Critical statement.
    invoke(primeReferralAction);

    expect(reporter.reportEvent).toHaveBeenCalledTimes(2);
  });

});
