import promiseMiddleware from 'core/store/middleware/promise';


describe('Redux Promise middleware', () => {

  function setUp() {
    const store = {
      getState: jest.fn(() => ({})),
      dispatch: jest.fn(),
    };
    const next = jest.fn();

    const invoke = (action) => promiseMiddleware(store)(next)(action);

    return { store, next, invoke };
  };

  it('allows actions with non-Promise type payloads to pass through.', () => {
    const { store, next, invoke } = setUp();
    const action = { type: 'TEST' };

    // Critical statement.
    invoke(action);

    expect(store.dispatch).toHaveBeenCalledTimes(0)
    expect(next).toHaveBeenCalledWith(action);
  });

  it('handles an action with a Promise payload and dispatch that payload\'s resolved value.', () => {
    const { store, next, invoke } = setUp();
    const tstType = 'TEST'
    const val = 'val'

    const initialAction = {
      type: tstType,
      payload: Promise.resolve(val)
    };
    const expectedSubsequentAction = {
      type: tstType,
      payload: val
    };

    // Critical statement.
    invoke(initialAction);

    return initialAction.payload.then(() => {
      expect(store.dispatch).toHaveBeenCalledWith(expectedSubsequentAction);
      expect(next).toHaveBeenCalled();
    });
  });

  it('handles an action with a Promise payload and dispatch that payload\'s rejected value.', () => {
    const { store, next, invoke } = setUp();
    const tstType = 'TEST'
    const val = 'val'

    const initialAction = {
      type: tstType,
      payload: Promise.reject(val)
    };
    const expectedSubsequentAction = {
      type: tstType,
      error: true,
      payload: new Error(val)
    };

    // Critical statement.
    invoke(initialAction);

    return initialAction.payload.then(() => {
      expect(true).toBeFalsy(); // should not reach here.
    }, () => {
      expect(store.dispatch).toHaveBeenCalledWith(expectedSubsequentAction);
      expect(next).toHaveBeenCalled();
    });
  });

});
