/* Manual Mocks - must mock before all other imports. */
import SCSClient from 'core/clients/CommunityServiceClient';

/* Helper methods used to craft mock responses */
const getBlob = (data) => {
  return new Blob([JSON.stringify(data, null, 2)], { type: 'application/json' });
}

const getDefaultData = () => {
  return { some: 'stuff' };
}

const getSumoWhitelistedData = () => {
  return {extensionData: {experiments: ["SUMO"]}};
}

const getSumoNotWhitelistedData = () => {
  return {extensionData: {experiments: ["NOTSUMO"]}};
}

const getDefaultDataBlob = () => {
  const data = getDefaultData();
  return getBlob(data);
};

const getDefaultResponse = () => {
  const blob = getDefaultDataBlob();
  return new Response(blob, {status: 200});
};

/* Mocked fetch requests. */
const mockFetchRequestFailed = jest.fn().mockImplementation(() => {
  return Promise.reject();
});

const mockFetchRequestBadResponse = jest.fn().mockImplementation(() => {
  const blob = getDefaultDataBlob();
  const response = new Response(blob, {status: 400});
  return new Promise(resolve => resolve(response));
});

const mockFetchRequestBadJson = jest.fn().mockImplementation(() => {
  const response = new Response(null, {status: 200});
  return new Promise(resolve => resolve(response));
});

const mockFetchRequestValidResponse = jest.fn().mockImplementation(() => {
  const blob = getDefaultDataBlob();
  const response = new Response(blob, {status: 200});
  return new Promise(resolve => resolve(response));
});

const mockFetchRequestSumoWhitelisted = jest.fn().mockImplementation(() => {
  const data = getSumoWhitelistedData();
  const blob = getBlob(data);
  const response = new Response(blob, {status: 200});
  return new Promise(resolve => resolve(response));
});

const mockFetchRequestSumoNotWhitelisted = jest.fn().mockImplementation(() => {
  const data = getSumoNotWhitelistedData();
  const blob = getBlob(data);
  const response = new Response(blob, {status: 200});
  return new Promise(resolve => resolve(response));
});

/* Default mocks, set up as to not accidentally allow real requests to go through. */
SCSClient.GetOverlayExtension = mockFetchRequestValidResponse;
SCSClient.SpendSubscriptionCredit = mockFetchRequestValidResponse;
SCSClient.GetOverlayExtensionLocation = mockFetchRequestValidResponse;
SCSClient.SetOverlayExtensionLocation = mockFetchRequestValidResponse;

/* Imports */
import configureStore from 'redux-mock-store';
import middleware from 'core/store/middleware';
import * as communityServiceTypes from 'core/store/modules/CommunityService/constants';
import * as errorMessageTypes from 'core/store/modules/ErrorMessage/constants';
import * as actions from 'core/store/modules/CommunityService/actions';

/* Tests */
describe('CommunityService module actions', () => {
  describe('creators', () => {
    it('creates a thunk which dispatches an action with an error payload when fetchOverlayExtension is called and the promise returned by fetch is rejected.', async () => {
      SCSClient.GetOverlayExtension = mockFetchRequestFailed;

      const jwt = 'JWT';
      const locale = 'LOCALE';
      const dateOverride = '2018-04-05T00:00:00Z';
      const expectedErrorMessage = new Error('fetchOverlayExtension encountered an error: undefined');
      const callbackIfWhitelistedForSumo = jest.fn();
      const dispatch = jest.fn();
      
      await actions.fetchOverlayExtension({jwt, locale, dateOverride, callbackIfWhitelistedForSumo})(dispatch);

      expect(dispatch).toBeCalledWith({type: communityServiceTypes.FETCH_OVERLAY_EXTENSION, payload: expectedErrorMessage, error: true});
      expect(callbackIfWhitelistedForSumo).toHaveBeenCalledTimes(0);
    });

    it('creates a thunk which dispatches an action with an error payload when fetchOverlayExtension is called and the response is not ok.', async () => {
      SCSClient.GetOverlayExtension = mockFetchRequestBadResponse;

      const jwt = 'JWT';
      const locale = 'LOCALE';
      const dateOverride = '2018-04-05T00:00:00Z';
      const expectedErrorMessage = new Error('fetchOverlayExtension encountered an error: undefined');
      const callbackIfWhitelistedForSumo = jest.fn();
      const dispatch = jest.fn();
      
      await actions.fetchOverlayExtension({jwt, locale, dateOverride, callbackIfWhitelistedForSumo})(dispatch);

      expect(dispatch).toBeCalledWith({type: communityServiceTypes.FETCH_OVERLAY_EXTENSION, payload: expectedErrorMessage, error: true});
      expect(callbackIfWhitelistedForSumo).toHaveBeenCalledTimes(0);
    });

    it('creates a thunk which dispatches an action with an error payload when fetchOverlayExtension is called and the response contains bad JSON.', async () => {
      SCSClient.GetOverlayExtension = mockFetchRequestBadJson;

      const jwt = 'JWT';
      const locale = 'LOCALE';
      const dateOverride = '2018-04-05T00:00:00Z';
      const expectedErrorMessage = new Error('fetchOverlayExtension encountered an error: SyntaxError: Unexpected end of JSON input');
      const callbackIfWhitelistedForSumo = jest.fn();
      const dispatch = jest.fn();
      
      await actions.fetchOverlayExtension({jwt, locale, dateOverride, callbackIfWhitelistedForSumo})(dispatch);

      expect(dispatch).toBeCalledWith({type: communityServiceTypes.FETCH_OVERLAY_EXTENSION, payload: expectedErrorMessage, error: true});
      expect(callbackIfWhitelistedForSumo).toHaveBeenCalledTimes(0);
    });

    it('creates a thunk which dispatches an action with a valid JSON payload when fetchOverlayExtension is called and the response contains valid JSON.', async () => {
      SCSClient.GetOverlayExtension = mockFetchRequestValidResponse;

      const jwt = 'JWT';
      const locale = 'LOCALE';
      const dateOverride = '2018-04-05T00:00:00Z';
      const callbackIfWhitelistedForSumo = jest.fn();
      const dispatch = jest.fn();
      
      await actions.fetchOverlayExtension({jwt, locale, dateOverride, callbackIfWhitelistedForSumo})(dispatch);

      expect(dispatch).toBeCalledWith({type: communityServiceTypes.FETCH_OVERLAY_EXTENSION, payload: getDefaultData()});
      expect(callbackIfWhitelistedForSumo).toHaveBeenCalledTimes(0);
    });

    it('creates a thunk which calls the provided callback when fetchOverlayExtension is called and the response contains valid JSON as well as contains the Sumo experiment.', async () => {
      SCSClient.GetOverlayExtension = mockFetchRequestSumoWhitelisted;

      const jwt = 'JWT';
      const locale = 'LOCALE';
      const dateOverride = '2018-04-05T00:00:00Z';
      const callbackIfWhitelistedForSumo = jest.fn();
      const dispatch = jest.fn();
      
      await actions.fetchOverlayExtension({jwt, locale, dateOverride, callbackIfWhitelistedForSumo})(dispatch);

      expect(dispatch).toBeCalledWith({type: communityServiceTypes.FETCH_OVERLAY_EXTENSION, payload: getSumoWhitelistedData()});
      expect(callbackIfWhitelistedForSumo).toBeCalled();
    });

    it('creates a thunk which does not call the provided callback when fetchOverlayExtension is called and the response contains valid JSON but does not contain the Sumo experiment.', async () => {
      SCSClient.GetOverlayExtension = mockFetchRequestSumoNotWhitelisted;

      const jwt = 'JWT';
      const locale = 'LOCALE';
      const dateOverride = '2018-04-05T00:00:00Z';
      const callbackIfWhitelistedForSumo = jest.fn();
      const dispatch = jest.fn();
      
      await actions.fetchOverlayExtension({jwt, locale, dateOverride, callbackIfWhitelistedForSumo})(dispatch);

      expect(dispatch).toBeCalledWith({type: communityServiceTypes.FETCH_OVERLAY_EXTENSION, payload: getSumoNotWhitelistedData()});
      expect(callbackIfWhitelistedForSumo).toHaveBeenCalledTimes(0);
    });

    it('creates an action with a Promise payload that resolves to a Response when spendSubscriptionCredit is called.', () => {
      const jwt = 'JWT';

      const receivedAction = actions.spendSubscriptionCredit({ jwt });

      expect(SCSClient.SpendSubscriptionCredit).toHaveBeenCalled();
      expect(receivedAction.type).toBe(communityServiceTypes.SPEND_SUBSCRIPTION_CREDIT);
      expect(typeof receivedAction.payload).toBe('object');
      expect(typeof receivedAction.payload.then).toBe('function');
      expect(receivedAction.payload).resolves.toEqual(getDefaultResponse());
    });

    it('creates a thunk which dispatches an error action when getOverlayExtensionLocation is called and the promise returned by fetch is rejected.', async () => {
      SCSClient.GetOverlayExtensionLocation = mockFetchRequestFailed;
      const jwt = 'JWT';
      const dispatch = jest.fn();

      await actions.getOverlayExtensionLocation({jwt})(dispatch);
      
      expect(dispatch).toBeCalledWith({type: errorMessageTypes.SET_LIVE_CONFIG_ERROR_MESSAGE, payload: errorMessageTypes.getOverlayExtensionLocationErrorMessage});
    });

    it('creates a thunk which dispatches an error action when getOverlayExtensionLocation is called and the response is not ok.', async () => {
      SCSClient.GetOverlayExtensionLocation = mockFetchRequestBadResponse;
      const jwt = 'JWT';
      const dispatch = jest.fn();

      await actions.getOverlayExtensionLocation({jwt})(dispatch);
      
      expect(dispatch).toBeCalledWith({type: errorMessageTypes.SET_LIVE_CONFIG_ERROR_MESSAGE, payload: errorMessageTypes.getOverlayExtensionLocationErrorMessage});
    });

    it('creates a thunk which dispatches an error action when getOverlayExtensionLocation is called and the response contains bad JSON.', async () => {
      SCSClient.GetOverlayExtensionLocation = mockFetchRequestBadJson;
      const jwt = 'JWT';
      const dispatch = jest.fn();

      await actions.getOverlayExtensionLocation({jwt})(dispatch);
      
      expect(dispatch).toBeCalledWith({type: errorMessageTypes.SET_LIVE_CONFIG_ERROR_MESSAGE, payload: errorMessageTypes.getOverlayExtensionLocationErrorMessage});
    });

    it('creates a thunk which dispatches success actions when getOverlayExtensionLocation is called and a valid response is received.', async () => {
      SCSClient.GetOverlayExtensionLocation = mockFetchRequestValidResponse;
      const jwt = 'JWT';
      const dispatch = jest.fn();

      await actions.getOverlayExtensionLocation({jwt})(dispatch);
      
      expect(dispatch).toBeCalledWith({type: communityServiceTypes.GET_OVERLAY_EXTENSION_LOCATION, payload: getDefaultData()});
    });

    it('creates a thunk which dispatches an error action when setOverlayExtensionLocation is called and the promise returned by fetch is rejected.', async () => {
      SCSClient.SetOverlayExtensionLocation = mockFetchRequestFailed;
      const jwt = 'JWT';
      const location = 'LOCATION';
      const dispatch = jest.fn();

      await actions.setOverlayExtensionLocation({jwt, location})(dispatch);
      
      expect(dispatch).toBeCalledWith({type: errorMessageTypes.SET_LIVE_CONFIG_ERROR_MESSAGE, payload: errorMessageTypes.setOverlayExtensionLocationErrorMessage});
    });

    it('creates a thunk which dispatches an error action when setOverlayExtensionLocation is called and the response is not ok.', async () => {
      SCSClient.SetOverlayExtensionLocation = mockFetchRequestBadResponse;
      const jwt = 'JWT';
      const location = 'LOCATION';
      const dispatch = jest.fn();

      await actions.setOverlayExtensionLocation({jwt, location})(dispatch);
      
      expect(dispatch).toBeCalledWith({type: errorMessageTypes.SET_LIVE_CONFIG_ERROR_MESSAGE, payload: errorMessageTypes.setOverlayExtensionLocationErrorMessage});
    });

    it('creates a thunk which dispatches an error action when setOverlayExtensionLocation is called and the response contains bad JSON.', async () => {
      SCSClient.SetOverlayExtensionLocation = mockFetchRequestBadJson;
      const jwt = 'JWT';
      const location = 'LOCATION';
      const dispatch = jest.fn();

      await actions.setOverlayExtensionLocation({jwt, location})(dispatch);
      
      expect(dispatch).toBeCalledWith({type: errorMessageTypes.SET_LIVE_CONFIG_ERROR_MESSAGE, payload: errorMessageTypes.setOverlayExtensionLocationErrorMessage});
    });

    it('creates a thunk which dispatches success actions when setOverlayExtensionLocation is called and a valid response is received.', async () => {
      SCSClient.SetOverlayExtensionLocation = mockFetchRequestValidResponse;
      const jwt = 'JWT';
      const location = 'LOCATION';
      const dispatch = jest.fn();

      await actions.setOverlayExtensionLocation({jwt, location})(dispatch);
      
      expect(dispatch).toBeCalledWith({type: communityServiceTypes.SET_OVERLAY_EXTENSION_LOCATION, payload: getDefaultData()});
      expect(dispatch).toBeCalledWith({type: errorMessageTypes.CLEAR_LIVE_CONFIG_ERROR_MESSAGE});
    });
  });

  describe('store integration', () => {

    function setUp() {
      const mockStore = configureStore(middleware)({});
      return { mockStore };
    }

    /* Asynchronous */
    it('creates the correct SPEND_SUBSCRIPTION_CREDIT actions after executing spendSubscriptionCredit().', () => {
      const { mockStore } = setUp();
      const jwt = 'JWT';

      const isPromise = val => val && typeof val.then === 'function';

      // We need to read the response body, otherwise the expected second action will fail due to missing bodyUsed:true
      const bodyReadResponse = getDefaultResponse();
      bodyReadResponse.blob();

      const expectedFirstAction = { type: communityServiceTypes.SPEND_SUBSCRIPTION_CREDIT, payload: mockFetchRequestValidResponse() };
      const expectedSecondAction = { type: communityServiceTypes.SPEND_SUBSCRIPTION_CREDIT, payload: bodyReadResponse };
      const expectedThirdAction = { type: communityServiceTypes.SPEND_SUBSCRIPTION_CREDIT, payload: getDefaultResponse().json()};
      const expectedFourthAction = { type: communityServiceTypes.SPEND_SUBSCRIPTION_CREDIT, payload: getDefaultData() };

      // Critical Statements.
      const spendSubscriptionCreditAction = actions.spendSubscriptionCredit({ jwt });
      const dispatchedAction = mockStore.dispatch(spendSubscriptionCreditAction);
      
      let receivedActions = mockStore.getActions();
      expect(receivedActions).toContainEqual(expectedFirstAction);
      expect(isPromise(dispatchedAction.payload)).toBeTruthy();

      return dispatchedAction.payload.then(response => {
        receivedActions = mockStore.getActions();
        expect(receivedActions).toContainEqual(expectedSecondAction);
        expect(receivedActions).toContainEqual(expectedThirdAction);
        return response.clone().json() // clone so we can re-read the response.
      }).then(() => {
        receivedActions = mockStore.getActions();
        expect(receivedActions).toContainEqual(expectedFourthAction);
      });
    });
  });
});
