import { name } from "core/store/modules/CommunityService/constants";
import * as selectors from "core/store/modules/CommunityService/selectors";
import { LootType } from "core/types/CommunityService";
import { Location } from "core/types/CommunityService";

describe("CommunityService module selectors", () => {
  it("selects the CommunityService state.", () => {
    const expectedSelection = { foo: "bar" };

    const state = { [name]: expectedSelection };

    const receivedSelection = selectors.selectCommunityService(state);

    expect(receivedSelection).toEqual(expectedSelection);
  });

  it("returns isPrime=true if a user is prime", () => {
    const expectedSelection = true;
    const currentState = {
      user: {
        hasPrime: true,
      },
    };
    const state = { [name]: currentState };
    const receivedSelection = selectors.selectIsPrime(state);
    expect(receivedSelection).toEqual(expectedSelection);
  });

  it("returns isPrime=false if a user is not prime", () => {
    const expectedSelection = false;
    const currentState = {
      user: {
        hasPrime: false,
      },
    };
    const state = { [name]: currentState };
    const receivedSelection = selectors.selectIsPrime(state);
    expect(receivedSelection).toEqual(expectedSelection);
  });

  it("returns isPrime=false if there is missing data ", () => {
    const expectedSelection = false;

    let state = {};
    let receivedSelection = selectors.selectIsPrime(state);
    expect(receivedSelection).toEqual(expectedSelection);

    state = { [name]: {} };
    receivedSelection = selectors.selectIsPrime(state);
    expect(receivedSelection).toEqual(expectedSelection);

    let currentState = {
      user: {},
    };
    state = { [name]: currentState };
    receivedSelection = selectors.selectIsPrime(state);
    expect(receivedSelection).toEqual(expectedSelection);

    currentState = {
      user: {
        hasPrime: undefined,
      },
    };
    state = { [name]: currentState };
    receivedSelection = selectors.selectIsPrime(state);
    expect(receivedSelection).toEqual(expectedSelection);
  });

  it("returns hasCredit=true if a user is has not spent their subcredit", () => {
    const expectedSelection = true;
    const currentState = {
      user: {
        credit: {
          balance: 1,
        },
      },
    };
    const state = { [name]: currentState };
    const receivedSelection = selectors.selectHasCredit(state);
    expect(receivedSelection).toEqual(expectedSelection);
  });

  it("returns hasCredit=true if a user has more than one subcredit", () => {
    const expectedSelection = true;
    const currentState = {
      user: {
        credit: {
          balance: 2,
        },
      },
    };
    const state = { [name]: currentState };
    const receivedSelection = selectors.selectHasCredit(state);
    expect(receivedSelection).toEqual(expectedSelection);
  });

  it("returns hasCredit=false if a user is has spent their subcredit", () => {
    const expectedSelection = false;
    const currentState = {
      user: {
        credit: {
          balance: 0,
        },
      },
    };
    const state = { [name]: currentState };
    const receivedSelection = selectors.selectHasCredit(state);
    expect(receivedSelection).toEqual(expectedSelection);
  });

  it("returns hasCredit=false if there is missing data ", () => {
    const expectedSelection = false;

    let state = {};
    let receivedSelection = selectors.selectHasCredit(state);
    expect(receivedSelection).toEqual(expectedSelection);

    state = { [name]: {} };
    receivedSelection = selectors.selectHasCredit(state);
    expect(receivedSelection).toEqual(expectedSelection);

    let currentState = {
      user: {},
    };
    state = { [name]: currentState };
    receivedSelection = selectors.selectHasCredit(state);
    expect(receivedSelection).toEqual(expectedSelection);

    currentState = {
      user: {
        credit: {},
      },
    };
    state = { [name]: currentState };
    receivedSelection = selectors.selectHasCredit(state);
    expect(receivedSelection).toEqual(expectedSelection);

    currentState = {
      user: {
        credit: {
          balance: undefined,
        },
      },
    };
    state = { [name]: currentState };
    receivedSelection = selectors.selectHasCredit(state);
    expect(receivedSelection).toEqual(expectedSelection);
  });

  it("returns hasIdentitySharingEnabled=true if a user has granted access to the extension", () => {
    const expectedSelection = true;
    const currentState = {
      hasIdentitySharingEnabled: true,
    };
    const state = { [name]: currentState };
    const receivedSelection = selectors.selectHasIdentitySharingEnabled(state);
    expect(receivedSelection).toEqual(expectedSelection);
  });

  it("returns hasIdentitySharingEnabled=false if a user has revoked access to the extension", () => {
    const expectedSelection = false;
    const currentState = {
      hasIdentitySharingEnabled: false,
    };
    const state = { [name]: currentState };
    const receivedSelection = selectors.selectHasIdentitySharingEnabled(state);
    expect(receivedSelection).toEqual(expectedSelection);
  });

  it("returns hasIdentitySharingEnabled=false if there is missing data ", () => {
    const expectedSelection = false;

    let state = {};
    let receivedSelection = selectors.selectHasIdentitySharingEnabled(state);
    expect(receivedSelection).toEqual(expectedSelection);

    state = { [name]: {} };
    receivedSelection = selectors.selectHasIdentitySharingEnabled(state);
    expect(receivedSelection).toEqual(expectedSelection);
  });

  it("returns the content message if there is one", () => {
    const expectedSelection = "This is the best message ever";
    const currentState = {
      loot: {
        message: expectedSelection,
      },
    };
    const state = { [name]: currentState };
    const receivedSelection = selectors.selectContentMessage(state);
    expect(receivedSelection).toEqual(expectedSelection);
  });

  it("returns an empty string if there is no content message", () => {
    const expectedSelection = "";

    let state = {};
    let receivedSelection = selectors.selectContentMessage(state);
    expect(receivedSelection).toEqual(expectedSelection);

    state = { [name]: {} };
    receivedSelection = selectors.selectContentMessage(state);
    expect(receivedSelection).toEqual(expectedSelection);

    state = { [name]: { loot: {} } };
    receivedSelection = selectors.selectContentMessage(state);
    expect(receivedSelection).toEqual(expectedSelection);

    state = { [name]: { loot: { message: undefined } } };
    receivedSelection = selectors.selectContentMessage(state);
    expect(receivedSelection).toEqual(expectedSelection);
  });

  it("returns the content image if there is one", () => {
    const expectedSelection = "https://thisisanimage";
    const currentState = {
      loot: {
        imageUrl: expectedSelection,
      },
    };
    const state = { [name]: currentState };
    const receivedSelection = selectors.selectContentImage(state);
    expect(receivedSelection).toEqual(expectedSelection);
  });

  it("returns an empty string if there is no content image", () => {
    const expectedSelection = "";

    let state = {};
    let receivedSelection = selectors.selectContentImage(state);
    expect(receivedSelection).toEqual(expectedSelection);

    state = { [name]: {} };
    receivedSelection = selectors.selectContentImage(state);
    expect(receivedSelection).toEqual(expectedSelection);

    state = { [name]: { loot: {} } };
    receivedSelection = selectors.selectContentImage(state);
    expect(receivedSelection).toEqual(expectedSelection);

    state = { [name]: { loot: { imageUrl: undefined } } };
    receivedSelection = selectors.selectContentImage(state);
    expect(receivedSelection).toEqual(expectedSelection);
  });

  it("returns subscribed=true if a user is has spent their subcredit during this session", () => {
    const expectedSelection = true;
    const currentState = {
      user: {
        credit: {
          subscribed: true,
        },
      },
    };
    const state = { [name]: currentState };
    const receivedSelection = selectors.selectSubscribed(state);
    expect(receivedSelection).toEqual(expectedSelection);
  });

  it("returns subscribed=false if a user has not spent their subcredit during this session", () => {
    const expectedSelection = false;
    const currentState = {
      user: {
        credit: {
          subscribed: false,
        },
      },
    };
    const state = { [name]: currentState };
    const receivedSelection = selectors.selectSubscribed(state);
    expect(receivedSelection).toEqual(expectedSelection);
  });

  it("returns subscribed=false if there is missing data", () => {
    const expectedSelection = false;

    let state = {};
    let receivedSelection = selectors.selectSubscribed(state);
    expect(receivedSelection).toEqual(expectedSelection);

    state = { [name]: {} };
    receivedSelection = selectors.selectSubscribed(state);
    expect(receivedSelection).toEqual(expectedSelection);

    let currentState = {
      user: {},
    };
    state = { [name]: currentState };
    receivedSelection = selectors.selectSubscribed(state);
    expect(receivedSelection).toEqual(expectedSelection);

    currentState = {
      user: {
        credit: {},
      },
    };
    state = { [name]: currentState };
    receivedSelection = selectors.selectSubscribed(state);
    expect(receivedSelection).toEqual(expectedSelection);

    currentState = {
      user: {
        credit: {
          subscribed: undefined,
        },
      },
    };
    state = { [name]: currentState };
    receivedSelection = selectors.selectSubscribed(state);
    expect(receivedSelection).toEqual(expectedSelection);
  });

  it("returns the type of content", () => {
    const expectedSelection = LootType.GAME;
    const currentState = {
      loot: {
        type: expectedSelection,
      },
    };
    const state = { [name]: currentState };
    const receivedSelection = selectors.selectType(state);
    expect(receivedSelection).toEqual(expectedSelection);
  });

  it("returns 'LOOT' for the type if there is missing data", () => {
    const expectedSelection = LootType.LOOT;

    let state = {};
    let receivedSelection = selectors.selectType(state);
    expect(receivedSelection).toEqual(expectedSelection);

    state = { [name]: {} };
    receivedSelection = selectors.selectType(state);
    expect(receivedSelection).toEqual(expectedSelection);

    let currentState = {
      loot: {},
    };
    state = { [name]: currentState };
    receivedSelection = selectors.selectType(state);
    expect(receivedSelection).toEqual(expectedSelection);

    currentState = {
      loot: {
        type: "",
      },
    };
    state = { [name]: currentState };
    receivedSelection = selectors.selectType(state);
    expect(receivedSelection).toEqual(expectedSelection);
  });

  it("returns the signup destination link", () => {
    const expectedSelection = "www.amazon.com/signupfortwitchprim";

    const state = {
      [name]: {
        signup: {
          destinationLink: expectedSelection,
        },
      },
    };

    const receivedSelection = selectors.selectSignUpLink(state);
    expect(receivedSelection).toEqual(expectedSelection);

  });

  it("returns the offer destination link", () => {
    const expectedSelection = "offerlink.com";

    const state = {
      [name]: {
        loot: {
          destinationLink: expectedSelection,
        },
      },
    };

    const receivedSelection = selectors.selectOfferLink(state);
    expect(receivedSelection).toEqual(expectedSelection);
  });

  it("returns the user's current subscribed channel", () => {
    const expectedSelection = "turbochannelwhoa";

    const state = {
      [name]: {
        user: {
          lastSpend: {
            channel: expectedSelection
          },
        },
      },
    };

    const receivedSelection = selectors.selectSubscribedChannel(state);
    expect(receivedSelection).toEqual(expectedSelection);
  });

  it("returns the time of the user's subscription", () => {
    const expectedSelection = "5/12/2018";

    const state = {
      [name]: {
        user: {
          lastSpend: {
            timestamp: expectedSelection
          },
        },
      },
    };

    const receivedSelection = selectors.selectSubscribedTime(state);
    expect(receivedSelection).toEqual(expectedSelection);
  });

  it("returns the location of the user's extension", () => {
    const expectedSelection = Location.TOP_MIDDLE;
    

    const state = {
      [name]: {
        location: Location.TOP_MIDDLE,
      },
    };

    const receivedSelection = selectors.selectLocation(state);
    expect(receivedSelection).toEqual(expectedSelection);
  });
});
