/* Manual Mocks - must mock before all other imports. */
import TwitchClient from 'core/clients/TwitchClient';

const DATA = { some: 'stuff' };
const BLOB = new Blob([JSON.stringify(DATA, null, 2)], { type : 'application/json' });
const RESPONSE = new Response(BLOB, { status: 200, statusText: 'OK' });
const FETCH_RESPONSE = new Promise(resolve => resolve(RESPONSE));

const mockGetStreamerContext = jest.fn().mockImplementation(() => {
  return FETCH_RESPONSE;
});

TwitchClient.GetStreamerContext = mockGetStreamerContext;

/* Imports */
import configureStore from 'redux-mock-store';
import middleware from 'core/store/middleware';
import * as types from 'core/store/modules/Twitch/constants';
import * as actions from 'core/store/modules/Twitch/actions';

describe('Twitch module actions', () => {

  describe('creators', () => {
    it('creates an action with a Promise payload that resolves to a Response when getStreamerContext is called.', () => {
      const channelID = 'channelID';

      const receivedAction = actions.getStreamerContext({ channelID })

      expect(TwitchClient.GetStreamerContext).toHaveBeenCalled();
      expect(receivedAction.type).toBe(types.GET_STREAMER_CONTEXT);
      expect(typeof receivedAction.payload).toBe('object');
      expect(typeof receivedAction.payload.then).toBe('function');
      expect(receivedAction.payload).resolves.toEqual(RESPONSE);
    });
  });

  describe('store integration', () => {

    function setUp() {
      const mockStore = configureStore(middleware)({});

      return { mockStore };
    }

    beforeEach(() => {
      mockGetStreamerContext.mockClear();
    });

    /* Asynchronous */
    it('creates the correct GET_STREAMER_CONTEXT actions after executing getStreamerContext().', () => {
      const { mockStore } = setUp();
      const channelID = 'channelID';

      const isPromise = val => val && typeof val.then === 'function';

      const expectedFirstAction = { type: types.GET_STREAMER_CONTEXT, payload: FETCH_RESPONSE };
      const expectedSecondAction = { type: types.GET_STREAMER_CONTEXT, payload: RESPONSE };
      const expectedThirdAction = { type: types.GET_STREAMER_CONTEXT, payload: RESPONSE.clone().json() };
      const expectedFourthAction = { type: types.GET_STREAMER_CONTEXT, payload: DATA };

      // Critical Statements.
      const getStreamerContextAction = actions.getStreamerContext({ channelID });
      const dispatchedAction = mockStore.dispatch(getStreamerContextAction);

      let receivedActions = mockStore.getActions();
      expect(receivedActions).toContainEqual(expectedFirstAction);
      expect(isPromise(dispatchedAction.payload)).toBeTruthy();

      return dispatchedAction.payload.then(response => {
        receivedActions = mockStore.getActions();
        expect(receivedActions).toContainEqual(expectedSecondAction);
        expect(receivedActions).toContainEqual(expectedThirdAction);
        return response.clone().json() // clone so we can re-read the response.
      }).then(() => {
        receivedActions = mockStore.getActions();
        expect(receivedActions).toContainEqual(expectedFourthAction);
      });

    });
  });

});
