import { selectTwitchExtension, selectAuthorization, selectContext, selectOpaqueUserId, selectToken, selectTheme } from 'core/store/modules/TwitchExtension/selectors';
import { name, authorization, context } from 'core/store/modules/TwitchExtension/constants';

describe('TwitchExtension module selectors', () => {

  it('selectTwitchExtension selects the TwitchExtension state.', () => {
    const expectedSelection = {
      [authorization]: { foo: 'bar' },
      [context]: { some: 'stuff' }
    };

    const state = { [name]: expectedSelection };

    const receivedSelection = selectTwitchExtension(state);

    expect(receivedSelection).toEqual(expectedSelection);
  });

  it('selectAuthorization selects the TwitchExtension authorization state.', () => {
    const expectedSelection = { foo: 'bar' };

    const state = {
      [name]: {
        [authorization]: expectedSelection,
        [context]: { some: 'stuff' }
      }
    };

    const receivedSelection = selectAuthorization(state);

    expect(receivedSelection).toEqual(expectedSelection);
  });

  it('selectToken selects the jwt token.', () => {
    const expectedSelection = 'bar';

    const state = {
      [name]: {
        [authorization]: {token: expectedSelection},
        [context]: { some: 'stuff' }
      }
    };

    const receivedSelection = selectToken(state);

    expect(receivedSelection).toEqual(expectedSelection);
  });

  it('selectToken returns a blank string if the jwt token is missing.', () => {
    const state = {
      [name]: {
        [authorization]: {},
        [context]: { some: 'stuff' }
      }
    };

    const receivedSelection = selectToken(state);

    expect(receivedSelection).toEqual("");
  });

  it('selectContext selects the TwitchExtension context state.', () => {
    const expectedSelection = { some: 'stuff' };

    const state = {
      [name]: {
        [authorization]: { foo: 'bar' },
        [context]: expectedSelection
      }
    };

    const receivedSelection = selectContext(state);

    expect(receivedSelection).toEqual(expectedSelection);
  });

  it('selectOpaqueUserId returns the user id', () => {
    const expectedSelection = 'randomId';

    const state = {
      [name]: {
        [authorization]: { userId: expectedSelection }
      }
    };
    const receivedSelection = selectOpaqueUserId(state);
    expect(receivedSelection).toEqual(expectedSelection);
  });

  it('selectOpaqueUserId returns an empty string if data is missing', () => {
    const expectedSelection = '';
    let state = {};
    let receivedSelection = selectOpaqueUserId(state);
    expect(receivedSelection).toEqual(expectedSelection);

    state = {
      [name]: {}
    };
    receivedSelection = selectOpaqueUserId(state);
    expect(receivedSelection).toEqual(expectedSelection);

    state = {
      [name]: {
        [authorization]: {}
      }
    };
    receivedSelection = selectOpaqueUserId(state);
    expect(receivedSelection).toEqual(expectedSelection);
  });

  it('selectTheme returns the current theme', () => {
    const expectedSelection = 'dark';

    const state = {
      [name]: {
        [context]: { theme: expectedSelection }
      }
    };
    const receivedSelection = selectTheme(state);
    expect(receivedSelection).toEqual(expectedSelection);
  });

  it('selectTheme returns "default" if data is missing', () => {
    const expectedSelection = 'default';
    let state = {};
    let receivedSelection = selectTheme(state);
    expect(receivedSelection).toEqual(expectedSelection);

    state = {
      [name]: {}
    };
    receivedSelection = selectTheme(state);
    expect(receivedSelection).toEqual(expectedSelection);

    state = {
      [name]: {
        [context]: {}
      }
    };
    receivedSelection = selectTheme(state);
    expect(receivedSelection).toEqual(expectedSelection);
  });
});
