import * as React from "react";
import { COMPONENT, OVERLAY } from 'core/store/modules/Display/constants';
import { DismissButton } from "viewer/components/DismissButton";
import { DropdownButton } from "viewer/components/DropdownButton";
import { Location } from 'core/types/CommunityService'
import { RootComponent } from "viewer/Root/component";
import { shallow } from "enzyme";
import toJson from "enzyme-to-json";

interface PropOptions {
  communityServiceLoaded?: boolean;
  isDismissed?: boolean;
  isExpanded?: boolean;
  isPrime?: boolean;
  error?: boolean;
  typeOfExtension?: string;
  showGameLauncher?: boolean;
  reportGameLaunch?: () => {};
  location?: Location;
  displayAsSidenavPopup?: boolean;
}

const getDefaultProps = (options?: PropOptions) => {
  return (
    <RootComponent
        communityServiceLoaded={options && options.communityServiceLoaded || true}
        isDismissed={options && options.isDismissed}
        isExpanded={options && options.isExpanded}
        error={options && options.error}
        isPrime={options && options.isPrime}
        typeOfExtension={options && options.typeOfExtension || OVERLAY}
        showGameLauncher={options && options.showGameLauncher || true}
        reportGameLaunch={ options && options.reportGameLaunch || jest.fn() }
        location={options && options.location || Location.TOP_RIGHT}
        displayAsSidenavPopup={options && options.displayAsSidenavPopup || false}
      />
  );
}

const setupShallow = (options?: PropOptions) => {
  return shallow(getDefaultProps(options));
}

describe("Root presentation component", () => {
  it("renders null when dismissed", () => {
    const component = setupShallow({ isDismissed: true });

    component.setState({ show: true });

    const tree = toJson(component);
    expect(tree).toMatchSnapshot();
  });

  it("renders correctly when expanded", () => {
    const component = setupShallow({ isExpanded: true });

    component.setState({ show: true });

    const tree = toJson(component);
    expect(tree).toMatchSnapshot();
  });

  it("renders correctly as a sidenav popup", () => {
    const component = setupShallow({ displayAsSidenavPopup: true });

    expect(component.find('.prime-extension-sidenav-popup')).toHaveLength(1);
    expect(component.find('.user-watcher')).toHaveLength(0);
    expect(component.find(DismissButton)).toHaveLength(0);
    expect(component.find(DropdownButton)).toHaveLength(0);
  });

  it("renders correctly not as a sidenav popup", () => {
    const component = setupShallow({ displayAsSidenavPopup: false, isExpanded: true });

    component.setState({ show: true });

    expect(component.find('.prime-extension-sidenav-popup')).toHaveLength(0);
    expect(component.find('.user-watcher')).toHaveLength(1);
    expect(component.find(DismissButton)).toHaveLength(1);
    expect(component.find(DropdownButton)).toHaveLength(1);
  });

  it("renders correctly when expanded and the user is prime", () => {
    const component = setupShallow({ isExpanded: true, isPrime: true });

    component.setState({ show: true });

    const tree = toJson(component);
    expect(tree).toMatchSnapshot();
  });

  it("renders correctly when collapsed", () => {
    const component = setupShallow({ isExpanded: false });

    component.setState({ show: true });

    const tree = toJson(component);
    expect(tree).toMatchSnapshot();
  });

  it("does not disappear when collapsing after mousing out and back in", () => {
    const component = setupShallow({ isExpanded: true });

    component.setState({ show: true });

    component.simulate("mouseLeave");
    expect(component.state().show).toBeFalsy();

    component.simulate("mouseEnter");
    component.setProps({isExpanded: false});
    expect(component.state().show).toBeTruthy();
  });

  it("renders correctly when collapsed if there is an error", () => {
    const component = setupShallow({ error: true, isExpanded: false });

    component.setState({ show: true });

    const tree = toJson(component);
    expect(tree).toMatchSnapshot();
  });

  it("renders correctly when expanded if there is an error", () => {
    const component = setupShallow({ isExpanded: true, error: true });
    
    component.setState({ show: true });

    const tree = toJson(component);
    expect(tree).toMatchSnapshot();
  });

  it("renders correctly when the user is hovering and the extension is collapsed", () => {
    const component = setupShallow({ isExpanded: false, isPrime: false});

    component.setState({ show: true });

    component.find(".prime-extension-root").simulate("mouseMove", { stopPropagation: Function() });

    const tree = toJson(component);
    expect(tree).toMatchSnapshot();
  });

  it("renders correctly when expanded in component view and the user is prime", () => {
    const component = setupShallow({ isExpanded: true, isPrime: true, typeOfExtension: COMPONENT });

    component.setState({ show: true });

    const tree = toJson(component);
    expect(tree).toMatchSnapshot();
  });

  it("renders correctly in the top left", () => {
    const component = setupShallow({ location: Location.TOP_LEFT });

    component.setState({ show: true });

    const tree = toJson(component);
    expect(tree).toMatchSnapshot();
    expect(component.find('.prime-extension-root__extension').hasClass('prime-extension-root__extension--top-left')).toBeTruthy();
  })

  it("renders correctly in the top middle", () => {
    const component = setupShallow({ location: Location.TOP_MIDDLE });

    component.setState({ show: true });

    const tree = toJson(component);
    expect(tree).toMatchSnapshot();
    expect(component.find('.prime-extension-root__extension').hasClass('prime-extension-root__extension--top-middle')).toBeTruthy();
  })

  it("renders correctly in the top right", () => {
    const component = setupShallow({ location: Location.TOP_RIGHT });

    component.setState({ show: true });

    const tree = toJson(component);
    expect(tree).toMatchSnapshot();
    expect(component.find('.prime-extension-root__extension').hasClass('prime-extension-root__extension--top-right')).toBeTruthy();
  })

  it("renders correctly in the bottom left", () => {
    const component = setupShallow({ location: Location.BOTTOM_LEFT });

    component.setState({ show: true });

    const tree = toJson(component);
    expect(tree).toMatchSnapshot();
    expect(component.find('.prime-extension-root__extension').hasClass('prime-extension-root__extension--bottom-left')).toBeTruthy();
  })

  it("renders correctly in the bottom right", () => {
    const component = setupShallow({ location: Location.BOTTOM_RIGHT });
    
    component.setState({ show: true });

    const tree = toJson(component);
    expect(tree).toMatchSnapshot();
    expect(component.find('.prime-extension-root__extension').hasClass('prime-extension-root__extension--bottom-right')).toBeTruthy();
  })

  describe("extension game", () => {
    it("shows the icon in overlay mode", () => {
      const component = setupShallow({ isExpanded: true, isPrime: true });

      component.setState({ show: true });

      const tree = toJson(component);
      expect(tree).toMatchSnapshot();
      expect(component.find('.prime-extension-root__game__icon').length).toBe(1);
    });

    it("does not render icon in component mode", () => {
      const component = setupShallow({ isExpanded: true, isPrime: true, typeOfExtension: COMPONENT });
      
      component.setState({ show: true });

      const tree = toJson(component);
      expect(tree).toMatchSnapshot();
      expect(component.find('.prime-extension-root__game__icon').length).toBe(0);
    });

    it("does not render icon if the game is not enabled", () => {
      const component = setupShallow({ isExpanded: true, isPrime: true, showGameLauncher: false });

      const tree = toJson(component);
      expect(tree).toMatchSnapshot();
      expect(component.find('.prime-extension-root__game__icon').length).toBe(0);
    })

    it("shows the game after the icon was selected", () => {
      const component = setupShallow({ isExpanded: true, isPrime: true });

      component.setState({ showGame: true })

      const tree = toJson(component);
      expect(tree).toMatchSnapshot();
    });
  });

  describe("Auto-hide", () => {
    it("hides after one second when collapsed", () => {
      jest.useFakeTimers();
      const component = setupShallow();

      component.setState({ show: true });

      expect(setTimeout).toHaveBeenCalled();
      expect(component.state().show).toBeTruthy();

      jest.advanceTimersByTime(999);

      expect(component.state().show).toBeTruthy();

      jest.advanceTimersByTime(1);

      expect(component.state().show).toBeFalsy();
    });

    it("does not hide when expanded", () => {
      jest.useFakeTimers();
      const component = setupShallow({ isExpanded: true });

      component.setState({ show: true });

      expect(component.state().show).toBeTruthy();

      jest.runAllTimers();

      expect(component.state().show).toBeTruthy();
    });

    it("reappears on mouse move", () => {
      jest.useFakeTimers();
      const component = setupShallow();
      component.setState({ show: true });

      jest.runAllTimers();
      expect(component.state().show).toBeFalsy();

      // Critical
      component.simulate("mouseMove");

      expect(setTimeout).toHaveBeenCalled();
      expect(component.state().show).toBeTruthy();
    });

    it("hides on mouse leave", () => {
      const component = setupShallow();

      component.setState({ show: true });

      // Critical
      component.simulate("mouseLeave");

      expect(component.state().show).toBeFalsy();
    });
  });

})
