import { shallow } from "enzyme";
import toJson from "enzyme-to-json";
import * as MockDate from "mockdate";
import * as React from "react";
import { SUBSCRIBE_BUTTON, SUBSCRIBED_BANNER, SubscriptionComponent } from "viewer/components/Subscription/component";

interface OptionsProps {
  hasCredit?: boolean;
  canSubscribeToStreamer?: boolean;
  subscribed?: boolean;
  isSubscribing?: boolean;
  subscribedTime?: string;
  subscribedChannel?: string;
  onSubscribeClick?: () => void;
  setSubscribingStatus?: (status: boolean) => void;
}

const getSubscriptionComponent = (options?: OptionsProps) => {
  return (
    <SubscriptionComponent
        hasCredit={options && options.hasCredit}
        canSubscribeToStreamer={options && options.canSubscribeToStreamer}
        streamerName={ "streamerName" }
        onSubscribeClick={options && options.onSubscribeClick || jest.fn()}
        setSubscribingStatus={options && options.setSubscribingStatus || jest.fn()}
        subscribed={ options && options.subscribed }
        isSubscribing={ options && options.isSubscribing || false }
        subscribedTime={ options && options.subscribedTime || undefined }
        subscribedChannel={ options && options.subscribedChannel || undefined }
    />
  );
}

const setupShallow = (options?: OptionsProps) => {
  return shallow(getSubscriptionComponent(options));
}

document.getElementById = (id) => {
  return document.createElement("div");
};

describe("Subscription presentation component", () => {
  it("renders the name of the streamer and a subscribe button if the user has not used their free sub and the streamer is a partner/affiliate", () => {
    MockDate.set("5/12/2018");
    const component = setupShallow({ hasCredit: true, canSubscribeToStreamer: true });

    const tree = toJson(component);
    expect(tree).toMatchSnapshot();
  });

  it("renders nothing if the subscription is not ready and the user does not have an active subscription", () => {
    const component = setupShallow({ hasCredit: false, canSubscribeToStreamer: true });

    const tree = toJson(component);
    expect(tree).toMatchSnapshot();
  });

  it("renders nothing if the streamer cannot be subscribed to and the user does not have an active subscription", () => {
    const component = setupShallow({ hasCredit: true, canSubscribeToStreamer: false });

    const tree = toJson(component);
    expect(tree).toMatchSnapshot();
  });

  it("renders the subscribed view after a successful subscription", () => {
    MockDate.set("5/12/2018");
    const component = setupShallow({ hasCredit: false, canSubscribeToStreamer: true, subscribed: true, subscribedTime: "5/12/2018", subscribedChannel: "turbochannelsupreme"  });

    const tree = toJson(component);
    expect(tree).toMatchSnapshot();
  });

  it("calls onSubscribeClick and setSubscribingStatus when subscribing", () => {
    jest.useFakeTimers();
    const onSubscribeClick = jest.fn();
    const setSubscribingStatus = jest.fn();
    const component = setupShallow({ hasCredit: true, canSubscribeToStreamer: true, onSubscribeClick: onSubscribeClick, setSubscribingStatus: setSubscribingStatus });

    const subscribeButton = component.find({ "data-test-target": SUBSCRIBE_BUTTON });
    expect(subscribeButton).toHaveLength(1);
    subscribeButton.simulate("click");
    expect(onSubscribeClick).toBeCalled();
    expect(setSubscribingStatus).toBeCalled();
  });

  it("renders the user's current subscription status.", () => {
    const component = setupShallow({ hasCredit: false, canSubscribeToStreamer: true, subscribedTime: "5/12/2018", subscribedChannel: "turbochannelsupreme"});

    const tree = toJson(component);
    expect(tree).toMatchSnapshot();
  });

  it("renders the subscribe button if the user has a subscription, even if they are already subscribed to someone", () => {
    const component = setupShallow({ hasCredit: true, canSubscribeToStreamer: true, subscribedTime: "5/12/2018", subscribedChannel: "turbochannelsupreme" });

    const tree = toJson(component);
    expect(tree).toMatchSnapshot();
  });

  
  it("renders the subscribing animation if the user is currently subscribing", () => {
    const component = setupShallow({ hasCredit: true, canSubscribeToStreamer: true, isSubscribing: true, subscribedTime: "5/12/2018", subscribedChannel: "turbochannelsupreme" });

    const tree = toJson(component);
    expect(tree).toMatchSnapshot();
  });

  it("renders the user's current subscription status if the user has a credit but can't spend it on the streamer", () => {
    const component = setupShallow({ hasCredit: true, canSubscribeToStreamer: false, subscribedTime: "5/12/2018", subscribedChannel: "turbochannelsupreme" });

    const tree = toJson(component);
    expect(tree).toMatchSnapshot();
  });

});
