import GameClient from "core/clients/GameClient";
import { createAction } from "redux-actions";
import { END_GAME_SESSION, GET_GAME_SESSION, LEAVE_GAME, HANDLE_MATCHMAKING_UPDATE, GET_GAME_WHITELIST_STATUS, START_MATCHMAKING, START_GAME_SESSION, SET_DISPLAYED_WINNER, CANCEL_MATCHMAKING, UPDATE_LOCAL_CLOCK_TIME_DELTA } from "./constants";
import { endGameSessionErrorMessage, getGameSessionErrorMessage, joinGameSessionErrorMessage, leaveGameErrorMessage, startGameSessionErrorMessage, submitGameInputErrorMessage, setGameErrorMessage, clearGameErrorMessage, setLiveConfigErrorMessage, cancelMatchmakingErrorMessage } from "core/store/modules/ErrorMessage";

export const startGameSession = ({jwt}) => {
    return (dispatch) => {
        return GameClient.StartGameSession({jwt})
        .then((response) => response.ok ? dispatch(startGameSessionSuccess()) : Promise.reject())
        .catch(() => dispatch(setLiveConfigErrorMessage(startGameSessionErrorMessage)));
    }
}

export const endGameSession = ({jwt}) => {
    return (dispatch) => {
        return GameClient.EndGameSession({jwt})
        .then((response) => response.ok ? dispatch(endGameSessionSuccess()) : Promise.reject())
        .catch(() => dispatch(setLiveConfigErrorMessage(endGameSessionErrorMessage)));
    }
}

export const startMatchmaking = ({jwt}) => {
    return (dispatch) => {
        return GameClient.StartMatchmaking({jwt})
        .then((response) => { 
            if(response.ok) { 
                dispatch(startMatchmakingSuccess()); 
                dispatch(clearGameErrorMessage()) ;
            } 
            else { 
                return Promise.reject();
            }})
        .catch(() => { dispatch(setGameErrorMessage(joinGameSessionErrorMessage)); });
    }
}

export const getGameSession = ({jwt}) => {
    return (dispatch) => {
        const requestStartTime = Date.now();
        return GameClient.GetGameSession({jwt})
        .then((response) => response.ok ? response.json() : Promise.reject())
        .then((json) => {
            dispatch(getGameSessionSuccess(json)); 
            dispatch(clearGameErrorMessage());
            calculateNewTimeDelta(dispatch, requestStartTime, json.currentTimeOnServer);
        })
        .catch(() => { dispatch(setGameErrorMessage(getGameSessionErrorMessage)); dispatch(setLiveConfigErrorMessage(getGameSessionErrorMessage)); });
    }
}

export const submitGameInput = ({jwt, input}) => {
    return (dispatch) => {
        return GameClient.SubmitGameInput({jwt, input})
        .then((response) => { if (!response.ok) { return Promise.reject() }})
        .catch(() => dispatch(setGameErrorMessage(submitGameInputErrorMessage)));
    }
}

export const leaveGame = ({jwt}) => {
    return (dispatch) => {
        return GameClient.LeaveGame({jwt})
        .then((response) => {
            if (response.ok) {
                dispatch(leaveGameSuccess());
                dispatch(clearGameErrorMessage());
            } else {
                return Promise.reject();
            } 
        })
        .catch(() => dispatch(setGameErrorMessage(leaveGameErrorMessage)));
    }
}

export const cancelMatchmaking = ({jwt}) => {
    return (dispatch) => {
        return GameClient.CancelMatchmaking({jwt})
        .then((response) => {
            if (response.ok) {
                dispatch(cancelMatchmakingSuccess());
                dispatch(clearGameErrorMessage());
            } else {
                return Promise.reject();
            } 
        })
        .catch(() => dispatch(setGameErrorMessage(cancelMatchmakingErrorMessage)));
    }
}

export const handleMatchmakingUpdate = createAction(HANDLE_MATCHMAKING_UPDATE);
export const setDisplayedWinner = createAction(SET_DISPLAYED_WINNER);

const updateLocalClockTimeDelta = createAction(UPDATE_LOCAL_CLOCK_TIME_DELTA);
const getGameWhitelistStatusSuccess = createAction(GET_GAME_WHITELIST_STATUS);
const startGameSessionSuccess = createAction(START_GAME_SESSION);
const endGameSessionSuccess = createAction(END_GAME_SESSION);
const startMatchmakingSuccess = createAction(START_MATCHMAKING);
const getGameSessionSuccess = createAction(GET_GAME_SESSION);
const leaveGameSuccess = createAction(LEAVE_GAME);
const cancelMatchmakingSuccess = createAction(CANCEL_MATCHMAKING);

let lowestPingForTimeSync = null;
function calculateNewTimeDelta(dispatch, requestStartTime, currentTimeOnServer) {
    if (requestStartTime != null && currentTimeOnServer != null) {
        const requestEndTime = Date.now();
        const ping = requestEndTime - requestStartTime;
        if (lowestPingForTimeSync == null || ping <= lowestPingForTimeSync) {
            lowestPingForTimeSync = ping;
            const expectedTimeOnServer = requestStartTime + ping/2.0;
            const newTimeDelta = currentTimeOnServer - expectedTimeOnServer;
            dispatch(updateLocalClockTimeDelta(newTimeDelta));
        }
    }
}